#ifndef FILTER_H
#define FILTER_H

#include <vector>
#include <newmat.h>
#include "pppInclude.h"
#include "pppParlist.h"
#include "bnctime.h"
#include "t_prn.h"
#include "pppClient.h"

namespace BNC_PPP {

class t_pppParlist;
class t_pppObsPool;
class t_pppSatObs;

class t_pppFilter {
 public:
  t_pppFilter(t_pppObsPool* obsPool);
  ~t_pppFilter();

  t_irc processEpoch();

  t_irc datumTransformation();
  void initDatumTransformation(const std::vector<t_pppSatObs*>& allObs, bool pseudoObsIono);
  unsigned setTrafoObs();

  const ColumnVector&    x() const {return _xFlt;}
  const SymmetricMatrix& Q() const {return _QFlt;}

  int    numSat() const {return _numSat;}
  double HDOP() const {return _dop.H;}
  double HDOV() const {return _dop.V;}
  double PDOP() const {return _dop.P;}
  double GDOP() const {return _dop.G;}
  double trp() const {
    const std::vector<t_pppParam*>& par = _parlist->params();
    for (unsigned ii = 0; ii < par.size(); ++ii) {
      if (par[ii]->type() == t_pppParam::trp) {
        return x()[ii];
      }
    }
    return 0.0;
  };
  double trpStdev() const {
    const std::vector<t_pppParam*>& par = _parlist->params();
    for (unsigned ii = 0; ii < par.size(); ++ii) {
      if (par[ii]->type() == t_pppParam::trp) {
        return sqrt(Q()[ii][ii]);
      }
    }
    return 0.0;
  };

 private:
  class t_slip {
   public:
    t_slip() {
      _slip            = false;
      _obsSlipCounter  = -1;
      _biasJumpCounter = -1;
    }
    bool _slip;
    int  _obsSlipCounter;
    int  _biasJumpCounter;
  };

  class t_dop {
   public:
    t_dop() {reset();}
    void reset() {H = V = P = T = G = 0.0;}
    double H;
    double V;
    double P;
    double T;
    double G;
  };

  class t_datumTrafo {
  public:
    t_datumTrafo () {
      initIndices();
    }
    ~t_datumTrafo () {

    }

    void initIndices() {
      _firstRow = 1; _lastRow = 0;
    }

    void setFirstSystem(char firstSys) {
      _firstSys = firstSys;
    }

    bool firstSystem(char sys) {
      if (_firstSys == sys) {
        return true;
      }
      return false;
    }

    void updateIndices(char sys, int maxObsSys) {
      if (firstSystem(sys)) {
        initIndices();
      }
      else {
        _firstRow = _lastRow + 1;
      }
      _lastRow += maxObsSys;//LOG << "updateIndices: lastRow: " << _lastRow << endl;
    };

    void setNumObs(int maxObs) {_maxObs = maxObs;}
    void setNumPar(int numPar) { _numPar = numPar;}
    void setLastEpoParlist(t_pppParlist* parlist) {_parlist = parlist;}
    t_pppParlist* lastEpoParlist() {return _parlist;}
    int numPar() {return _numPar;}
    int numObs() {return _maxObs;}
    void updateNumObs() {//LOG << "updateObsNum _maxObs " << _maxObs;
      _maxObs = _lastRow;//LOG << "=>  _maxObs " << _maxObs << " _numPar: " << _numPar << endl;
      _AA1 = _AA1.SubMatrix(1, _lastRow, 1, _numPar);
      _AA2 = _AA2.SubMatrix(1, _lastRow, 1, _numPar);
    }

    const Matrix& AA1() {return _AA1;}
    const Matrix& AA2() {return _AA2;}
    const Matrix& D21() {return _D21;}

    void initAA() {//LOG << "initAA: _maxObs: " << _maxObs << " _numPar: " << _numPar << endl;
      _AA1.ReSize(_maxObs, _numPar); _AA1 = 0.0;
      _AA2.ReSize(_maxObs, _numPar); _AA2 = 0.0;
      _D21.ReSize(_numPar, _numPar); _D21 = 0.0;
    }

    void prepareAA(const Matrix& AA, int ind) {
      Matrix* Prep = &_AA2;
      if (ind == 1) {
        Prep = &_AA1;
      }
      Prep->SubMatrix(_firstRow, _lastRow, 1, _numPar) << AA;
    }

    void switchAA() {
      _AA1 = _AA2;
    }

    t_irc computeTrafoMatrix() {
      if (((_AA2.t() * _AA2)).Determinant() == 0.0) {
        LOG << "(_AA2.t() * _AA2).inv() is singular" << endl;
        return failure;
      }
      _D21 = ((_AA2.t() * _AA2).i()) * _AA2.t() * _AA1;
      return success;
    }

    void printMatrix(const Matrix& X, int nRow, int nCol) {
      for (int rr = 0; rr < nRow; rr++) {
        for (int cc = 0; cc < nCol; cc++) {
          LOG << setw(6) << setprecision(3) << X[rr][cc] << " ;";
        }
        LOG << endl;
      }
      LOG << endl;
    }
  private:
    int               _firstRow;
    int               _lastRow;
    Matrix            _AA1;
    Matrix            _AA2;
    Matrix            _D21;
    char              _firstSys;
    int               _maxObs;
    int               _numPar;
    QMap<char, t_prn> _refSatMapPseudoObs;
    t_pppParlist*     _parlist;
  };

  t_irc processSystem(const std::vector<t_lc::type>& LCs,
                      const std::vector<t_pppSatObs*>& obsVector,
                      const t_prn& refPrn,
                      bool pseudoObsIonoAvailable,
                      bool preProcessing);

  t_irc detectCycleSlips(const std::vector<t_lc::type>& LCs,
                         const std::vector<t_pppSatObs*>& obsVector,
                         const t_prn& refPrn,
                         bool preProcessing);

  t_irc resetAmb(t_prn prn, const std::vector<t_pppSatObs*>& obsVector,
                 SymmetricMatrix* QSav = 0, ColumnVector* xSav = 0);

  void cmpDOP(const std::vector<t_pppSatObs*>& obsVector);

  void predictCovCrdPart(const SymmetricMatrix& QFltOld);

  t_irc addNoiseToIono(char sys);

  bool resetRefSatellitesLastEpoch(std::vector<t_pppSatObs*>& obsVector);

  bncTime         _epoTime;
  t_pppParlist*   _parlist;
  t_pppObsPool*   _obsPool;
  t_datumTrafo*   _datumTrafo;
  SymmetricMatrix _QFlt;
  ColumnVector    _xFlt;
  ColumnVector    _x0;
  t_slip          _slips[t_prn::MAXPRN+1];
  int             _numSat;
  t_dop           _dop;
  bncTime         _firstEpoTime;
  bncTime         _lastEpoTimeOK;
  t_prn           _refPrn;
};

}

#endif
