#ifndef RTCM3_CLOCK_ORBIT_IGS_H
#define RTCM3_CLOCK_ORBIT_IGS_H

/* Programheader

        Name:           clock_orbit_igs.h
        Project:        RTCM3
        Version:        $Id: clock_orbit_igs.h 8966 2020-07-01 07:48:35Z stuerze $
        Authors:        Dirk Stöcker, Andrea Stürze
        Description:    state space approach: IGS
*/

#include <string.h>

enum SatelliteReferenceDatum {
  DATUM_ITRF  = 0,
  DATUM_LOCAL = 1
};

enum IGS_NUMBERS {
 RTCM_MESSAGE_NUMBER_IGS = 4076,
 IGS_SSR_VERSION         = 3
};

enum COR_BASE {// Orbit Correction
  COBBASE_GPS     =  21,
  COBBASE_GLONASS =  41,
  COBBASE_GALILEO =  61,
  COBBASE_QZSS    =  81,
  COBBASE_BDS     = 101,
  COBBASE_SBAS    = 121,
  COBBASE_NUM
};

enum COR_OFFSET {   // sub-type message, for example:
  COBOFS_ORBIT = 0, // GPS: IM21
  COBOFS_CLOCK,     // GPS: IM22
  COBOFS_COMBINED,  // GPS: IM23
  COBOFS_HR,        // GPS: IM24
  COBOFS_CBIAS,     // GPS: IM25
  COBOFS_PBIAS,     // GPS: IM26
  COBOFS_URA,       // GPS: IM27
  COBOFS_NUM
};

enum ClockOrbitType {
  COTYPE_GPSORBIT        = COBBASE_GPS     + COBOFS_ORBIT,
  COTYPE_GPSCLOCK,
  COTYPE_GPSCOMBINED     = COBBASE_GPS     + COBOFS_COMBINED,
  COTYPE_GPSHR,
  COTYPE_GPSURA          = COBBASE_GPS     + COBOFS_URA,

  COTYPE_GLONASSORBIT    = COBBASE_GLONASS + COBOFS_ORBIT,
  COTYPE_GLONASSCLOCK,
  COTYPE_GLONASSCOMBINED = COBBASE_GLONASS + COBOFS_COMBINED,
  COTYPE_GLONASSHR,
  COTYPE_GLONASSURA      = COBBASE_GLONASS + COBOFS_URA,

  COTYPE_GALILEOORBIT    = COBBASE_GALILEO + COBOFS_ORBIT,
  COTYPE_GALILEOCLOCK,
  COTYPE_GALILEOCOMBINED = COBBASE_GALILEO + COBOFS_COMBINED,
  COTYPE_GALILEOHR,
  COTYPE_GALILEOURA      = COBBASE_GALILEO + COBOFS_URA,

  COTYPE_QZSSORBIT       = COBBASE_QZSS    + COBOFS_ORBIT,
  COTYPE_QZSSCLOCK,
  COTYPE_QZSSCOMBINED    = COBBASE_QZSS    + COBOFS_COMBINED,
  COTYPE_QZSSHR,
  COTYPE_QZSSURA         = COBBASE_QZSS    + COBOFS_URA,

  COTYPE_BDSORBIT        = COBBASE_BDS     + COBOFS_ORBIT,
  COTYPE_BDSCLOCK,
  COTYPE_BDSCOMBINED     = COBBASE_BDS     + COBOFS_COMBINED,
  COTYPE_BDSHR,
  COTYPE_BDSURA          = COBBASE_BDS     + COBOFS_URA,

  COTYPE_SBASORBIT       = COBBASE_SBAS    + COBOFS_ORBIT,
  COTYPE_SBASCLOCK,
  COTYPE_SBASCOMBINED    = COBBASE_SBAS    + COBOFS_COMBINED,
  COTYPE_SBASHR,
  COTYPE_SBASURA         = COBBASE_SBAS    + COBOFS_URA,

  COTYPE_AUTO = 0,
};

enum CodeBiasType {
  CBTYPE_GPS     = COBBASE_GPS     + COBOFS_CBIAS,
  CBTYPE_GLONASS = COBBASE_GLONASS + COBOFS_CBIAS,
  CBTYPE_GALILEO = COBBASE_GALILEO + COBOFS_CBIAS,
  CBTYPE_QZSS    = COBBASE_QZSS    + COBOFS_CBIAS,
  CBTYPE_BDS     = COBBASE_BDS     + COBOFS_CBIAS,
  CBTYPE_SBAS    = COBBASE_SBAS    + COBOFS_CBIAS,
  CBTYPE_AUTO    = 0
};

enum PhaseBiasType {
  PBTYPE_GPS     = COBBASE_GPS     + COBOFS_PBIAS,
  PBTYPE_GLONASS = COBBASE_GLONASS + COBOFS_PBIAS,
  PBTYPE_GALILEO = COBBASE_GALILEO + COBOFS_PBIAS,
  PBTYPE_QZSS    = COBBASE_QZSS    + COBOFS_PBIAS,
  PBTYPE_BDS     = COBBASE_BDS     + COBOFS_PBIAS,
  PBTYPE_SBAS    = COBBASE_SBAS    + COBOFS_PBIAS,
  PBTYPE_AUTO    = 0
};

enum VTECType {
  VTEC_BASE = 201
};

/* if some systems aren't supported at all, change the following numbers to zero
for these systems to save space */
enum COR_CONSTANTS {
  CLOCKORBIT_BUFFERSIZE    = 8192,
  CLOCKORBIT_NUMGPS        =   32,
  CLOCKORBIT_NUMGLONASS    =   26,
  CLOCKORBIT_NUMGALILEO    =   36,
  CLOCKORBIT_NUMQZSS       =   10,
  CLOCKORBIT_NUMBDS        =   65,
  CLOCKORBIT_NUMSBAS       =   38,
  CLOCKORBIT_NUMBIAS       =  100,
  CLOCKORBIT_NUMIONOLAYERS =    4,
  CLOCKORBIT_MAXIONOORDER  =   16,
  CLOCKORBIT_MAXIONODEGREE =   16
};

enum COR_SATSYSTEM {
  CLOCKORBIT_SATGPS=0,
  CLOCKORBIT_SATGLONASS,
  CLOCKORBIT_SATGALILEO,
  CLOCKORBIT_SATQZSS,
  CLOCKORBIT_SATBDS,
  CLOCKORBIT_SATSBAS,
  CLOCKORBIT_SATNUM
};

enum COR_OFFSETS {
  CLOCKORBIT_OFFSETGPS     = 0,
  CLOCKORBIT_OFFSETGLONASS = CLOCKORBIT_NUMGPS,
  CLOCKORBIT_OFFSETGALILEO = CLOCKORBIT_NUMGPS  + CLOCKORBIT_NUMGLONASS,
  CLOCKORBIT_OFFSETQZSS    = CLOCKORBIT_NUMGPS  + CLOCKORBIT_NUMGLONASS + CLOCKORBIT_NUMGALILEO,
  CLOCKORBIT_OFFSETBDS     = CLOCKORBIT_NUMGPS  + CLOCKORBIT_NUMGLONASS + CLOCKORBIT_NUMGALILEO
                           + CLOCKORBIT_NUMQZSS,
  CLOCKORBIT_OFFSETSBAS    = CLOCKORBIT_NUMGPS  + CLOCKORBIT_NUMGLONASS + CLOCKORBIT_NUMGALILEO
                           + CLOCKORBIT_NUMQZSS + CLOCKORBIT_NUMBDS,
  CLOCKORBIT_COUNTSAT      = CLOCKORBIT_NUMGPS  + CLOCKORBIT_NUMGLONASS + CLOCKORBIT_NUMGALILEO
                           + CLOCKORBIT_NUMSBAS + CLOCKORBIT_NUMQZSS    + CLOCKORBIT_NUMBDS
};

enum CodeType {
  CODETYPEGPS_L1_CA          = 0,
  CODETYPEGPS_L1_P           = 1,
  CODETYPEGPS_L1_Z           = 2,
  CODETYPEGPS_L1C_D          = 3,
  CODETYPEGPS_L1C_P          = 4,
  CODETYPEGPS_L2_CA          = 5,
  CODETYPEGPS_SEMI_CODELESS  = 6,
  CODETYPEGPS_L2C_M          = 7,
  CODETYPEGPS_L2C_L          = 8,
  //RESERVED                 = 9,
  CODETYPEGPS_L2_P           = 10,
  CODETYPEGPS_L2_Z           = 11,
  //RESERVED                 = 12,
  //RESERVED                 = 13,
  CODETYPEGPS_L5_I           = 14,
  CODETYPEGPS_L5_Q           = 15,

  CODETYPEGLONASS_L1_CA      = 0,
  CODETYPEGLONASS_L1_P       = 1,
  CODETYPEGLONASS_L2_CA      = 2,
  CODETYPEGLONASS_L2_P       = 3,
  CODETYPEGLONASS_L1a_OCd    = 4,
  CODETYPEGLONASS_L1a_OCp    = 5,
  CODETYPEGLONASS_L2a_CSI    = 6,
  CODETYPEGLONASS_L2a_OCp    = 7,
  CODETYPEGLONASS_L3_I       = 8,
  CODETYPEGLONASS_L3_Q       = 9,

  CODETYPEGALILEO_E1_A       = 0,
  CODETYPEGALILEO_E1_B       = 1,
  CODETYPEGALILEO_E1_C       = 2,
  //RESERVED_E1_BC           = 3,
  //RESERVED_E1_ABC          = 4,
  CODETYPEGALILEO_E5A_I      = 5,
  CODETYPEGALILEO_E5A_Q      = 6,
  //RESERVED_E5A_IQ          = 7,
  CODETYPEGALILEO_E5B_I      = 8,
  CODETYPEGALILEO_E5B_Q      = 9,
  //RESERVED_E5B_IQ          = 10,
  //RESERVED_E5_I            = 11,
  //RESERVED_E5_Q            = 12,
  //RESERVED_E5_IQ           = 13,
  CODETYPEGALILEO_E6_A       = 14,
  CODETYPEGALILEO_E6_B       = 15,
  CODETYPEGALILEO_E6_C       = 16,
  //RESERVED_E6_BC           = 17,
  //RESERVED_E6_ABC          = 18,

  CODETYPEQZSS_L1_CA         = 0,
  CODETYPEQZSS_L1C_D         = 1,
  CODETYPEQZSS_L1C_P         = 2,
  CODETYPEQZSS_L2C_M         = 3,
  CODETYPEQZSS_L2C_L         = 4,
  //RESEVED_L2C_ML           = 5,
  CODETYPEQZSS_L5_I          = 6,
  CODETYPEQZSS_L5_Q          = 7,
  //RESERVED_L5_IQ           = 8,
  CODETYPEQZSS_L6_D          = 9,
  CODETYPEQZSS_L6_P          = 10,
  //RESERVED_L6_DP           = 11,
  //RESERVED_L1C_DP          = 12,
  //RESERVED_L1_S            = 13,
  //RESERVED_L5_D            = 14,
  //RESERVED_L5_P            = 15,
  //RESERVED_L5_DP           = 16,
  CODETYPEQZSS_L6_E          = 17,
  //RESERVED_L6_DE           = 18,

  CODETYPE_BDS_B1_I          = 0,
  CODETYPE_BDS_B1_Q          = 1,
  //RESERVED_BDS_B1_IQ       = 2,
  CODETYPE_BDS_B3_I          = 3,
  CODETYPE_BDS_B3_Q          = 4,
  //RESERVED_BDS_B3_IQ       = 5,
  CODETYPE_BDS_B2_I          = 6,
  CODETYPE_BDS_B2_Q          = 7,
  //RESERVED_BDS_B2_IQ       = 8,
  CODETYPE_BDS_B1a_D         = 9,
  CODETYPE_BDS_B1a_P         = 10,
  //RESERVED_BDS_B1a_DP      = 11,
  CODETYPE_BDS_B2a_D         = 12,
  CODETYPE_BDS_B2a_P         = 13,
  //RESEVED_BDS_B2a_DP       = 14,
  CODETYPE_BDS_B1_A          = 15,//NEW 1A
  //RESERVED                 = 16,
  //RESERVED                 = 17,
  CODETYPE_BDS_B3_A          = 18,//NEW 6A

  CODETYPE_SBAS_L1_CA        = 0,
  CODETYPE_SBAS_L5_I         = 1,
  CODETYPE_SBAS_L5_Q         = 2
  //RESERVED_SBAS_L5_IQ      = 3

};

#define SSR_MAXURA 5.5 /* > 5466.5mm in meter */

/* satellite system data is stored with offset CLOCKORBIT_OFFSET...
in the data structures. So first GLONASS satellite is at
xxx->Sat[CLOCKORBIT_OFFSETGLONASS], first GPS satellite is
xxx->Sat[CLOCKORBIT_OFFSETGPS]. */

struct ClockOrbit {
  enum ClockOrbitType messageType;
  unsigned int EpochTime[CLOCKORBIT_SATNUM];   /* 0 .. system specific maximum */
  unsigned int NumberOfSat[CLOCKORBIT_SATNUM]; /* 0 .. CLOCKORBIT_NUM... */
  unsigned int Supplied[COBOFS_NUM];           /* boolean */
  unsigned int SSRIOD;
  unsigned int SSRProviderID;
  unsigned int SSRSolutionID;
  unsigned int UpdateInterval;
  enum SatelliteReferenceDatum SatRefDatum;
  struct SatData {
    unsigned int ID; /* all */
    unsigned int IOD; /* all */
    unsigned int toe; /* SBAS, BDS */
    double UserRangeAccuracy; /* accuracy values in [m] */
    double hrclock;
    struct OrbitPart {
      double DeltaRadial;           /* m */
      double DeltaAlongTrack;       /* m */
      double DeltaCrossTrack;       /* m */
      double DotDeltaRadial;        /* m/s */
      double DotDeltaAlongTrack;    /* m/s */
      double DotDeltaCrossTrack;    /* m/s */
    } Orbit;
    struct ClockPart {
      double DeltaA0;               /* m */
      double DeltaA1;               /* m/s */
      double DeltaA2;               /* m/ss */
    } Clock;
  } Sat[CLOCKORBIT_COUNTSAT];
};

struct CodeBias {
  enum CodeBiasType messageType;
  unsigned int EpochTime[CLOCKORBIT_SATNUM];   /* 0 .. system specific maximum */
  unsigned int NumberOfSat[CLOCKORBIT_SATNUM]; /* 0 .. CLOCKORBIT_NUM... */
  unsigned int UpdateInterval;
  unsigned int SSRIOD;
  unsigned int SSRProviderID;
  unsigned int SSRSolutionID;
  struct BiasSat {
    unsigned int ID; /* all */
    unsigned int NumberOfCodeBiases;
    struct CodeBiasEntry {
      enum CodeType Type;
      float         Bias;           /* m */
    } Biases[CLOCKORBIT_NUMBIAS];
  } Sat[CLOCKORBIT_COUNTSAT];
};

struct PhaseBias {
  enum PhaseBiasType messageType;
  unsigned int EpochTime[CLOCKORBIT_SATNUM];   /* 0 .. system specific maximum */
  unsigned int NumberOfSat[CLOCKORBIT_SATNUM]; /* 0 .. CLOCKORBIT_NUM... */
  unsigned int UpdateInterval;
  unsigned int SSRIOD;
  unsigned int SSRProviderID;
  unsigned int SSRSolutionID;
  unsigned int DispersiveBiasConsistencyIndicator;
  unsigned int MWConsistencyIndicator;
  struct PhaseBiasSat {
    unsigned int ID; /* all */
    unsigned int NumberOfPhaseBiases;
    double YawAngle; /* radiant */
    double YawRate;  /* radiant/s */
    struct PhaseBiasEntry {
      enum CodeType Type;
      unsigned int  SignalIntegerIndicator;
      unsigned int  SignalsWideLaneIntegerIndicator;
      unsigned int  SignalDiscontinuityCounter;
      float         Bias;           /* m */
    } Biases[CLOCKORBIT_NUMBIAS];
  } Sat[CLOCKORBIT_COUNTSAT];
};

struct VTEC {
  unsigned int EpochTime; /* GPS */
  unsigned int UpdateInterval;
  unsigned int SSRIOD;
  unsigned int SSRProviderID;
  unsigned int SSRSolutionID;
  unsigned int NumLayers; /* 1-4 */
  double Quality;
  struct IonoLayers {
    double       Height; /* m */
    unsigned int Degree; /* 1-16 */
    unsigned int Order; /* 1-16 */
    double       Sinus[CLOCKORBIT_MAXIONODEGREE][CLOCKORBIT_MAXIONOORDER];
    double       Cosinus[CLOCKORBIT_MAXIONODEGREE][CLOCKORBIT_MAXIONOORDER];
  } Layers[CLOCKORBIT_NUMIONOLAYERS];
};

/* return size of resulting data or 0 in case of an error */
size_t MakeClockOrbit(const struct ClockOrbit *co, enum ClockOrbitType type,
       int moremessagesfollow, char *buffer, size_t size);
size_t MakeCodeBias(const struct CodeBias *b, enum CodeBiasType type,
       int moremessagesfollow, char *buffer, size_t size);
size_t MakePhaseBias(const struct PhaseBias *b, enum PhaseBiasType type,
       int moremessagesfollow, char *buffer, size_t size);
size_t MakeVTEC(const struct VTEC *b, int moremessagesfollow, char *buffer,
       size_t size);

enum GCOB_RETURN {
  /* all well */
  GCOBR_MESSAGEFOLLOWS        =  1,
  GCOBR_OK                    =  0,
  /* unknown data, a warning */
  GCOBR_UNKNOWNTYPE           = -1,
  GCOBR_UNKNOWNDATA           = -2,
  GCOBR_CRCMISMATCH           = -3,
  GCOBR_SHORTMESSAGE          = -4,
  /* failed to do the work */
  GCOBR_NOCLOCKORBITPARAMETER = -10,
  GCOBR_NOCODEBIASPARAMETER   = -11,
  GCOBR_NOPHASEBIASPARAMETER  = -12,
  GCOBR_NOVTECPARAMETER       = -13,
  /* data mismatch - data in storage does not match new data */
  GCOBR_TIMEMISMATCH          = -20,
  GCOBR_DATAMISMATCH          = -21,
  /* not enough data - can decode the block completely */
  GCOBR_SHORTBUFFER           = -30,
  GCOBR_MESSAGEEXCEEDSBUFFER  = -31};

/* NOTE: When an error message has been emitted, the output structures may have been modified. Make a copy of the previous variant before calling the
function to have a clean state. */

/* buffer should point to a RTCM3 block */
enum GCOB_RETURN GetSSR(struct ClockOrbit *co, struct CodeBias *b, struct VTEC *v,
       struct PhaseBias *pb, const char *buffer, size_t size, int *bytesused);

#endif /* RTCM3_CLOCK_ORBIT_IGS_H */
