// Part of BNC, a utility for retrieving decoding and
// converting GNSS data streams from NTRIP broadcasters.
//
// Copyright (C) 2007
// German Federal Agency for Cartography and Geodesy (BKG)
// http://www.bkg.bund.de
// Czech Technical University Prague, Department of Geodesy
// http://www.fsv.cvut.cz
//
// Email: euref-ip@bkg.bund.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation, version 2.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

/* -------------------------------------------------------------------------
 * BKG NTRIP Client
 * -------------------------------------------------------------------------
 *
 * Class:      t_pppSatObs
 *
 * Purpose:    Satellite observations
 *
 * Author:     L. Mervart
 *
 * Created:    29-Jul-2014
 *
 * Changes:    
 *
 * -----------------------------------------------------------------------*/


#include <iostream>
#include <cmath>
#include <newmatio.h>

#include "pppSatObs.h"
#include "bncconst.h"
#include "pppEphPool.h"
#include "pppStation.h"
#include "bncutils.h"
#include "bncantex.h"
#include "pppObsPool.h"
#include "pppClient.h"

using namespace BNC_PPP;
using namespace std;

// Constructor
////////////////////////////////////////////////////////////////////////////
t_pppSatObs::t_pppSatObs(const t_satObs& pppSatObs) {
  _prn     = pppSatObs._prn;
  _time    = pppSatObs._time;
  _outlier = false;
  _valid   = true;
  for (unsigned ii = 0; ii < t_frequency::max; ii++) {
    _obs[ii] = 0;
  }
  prepareObs(pppSatObs);
}

// Destructor
////////////////////////////////////////////////////////////////////////////
t_pppSatObs::~t_pppSatObs() {
  for (unsigned iFreq = 1; iFreq < t_frequency::max; iFreq++) {
    delete _obs[iFreq];
  }
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppSatObs::prepareObs(const t_satObs& pppSatObs) {

  _model.reset();

  // Select pseudoranges and phase observations
  // ------------------------------------------
  const string preferredAttrib = "CWP_";

  for (unsigned iFreq = 1; iFreq < t_frequency::max; iFreq++) {
    string frqNum = t_frequency::toString(t_frequency::type(iFreq)).substr(1);
    for (unsigned iPref = 0; iPref < preferredAttrib.length(); iPref++) {
      string obsType = (preferredAttrib[iPref] == '_') ? frqNum : frqNum + preferredAttrib[iPref];
      if (_obs[iFreq] == 0) {
        for (unsigned ii = 0; ii < pppSatObs._obs.size(); ii++) {
          const t_frqObs* obs = pppSatObs._obs[ii];
          if (obs->_rnxType2ch == obsType && obs->_codeValid && obs->_phaseValid) {
            _obs[iFreq] = new t_frqObs(*obs);
            _obs[iFreq]->_freqType = t_frequency::type(iFreq);
          }
        }
      }
    }
  }

  // Used frequency types
  // --------------------
  _fType1 = t_lc::toFreq(_prn.system(),t_lc::l1);
  _fType2 = t_lc::toFreq(_prn.system(),t_lc::l2);

  // Check whether all required frequencies available
  // ------------------------------------------------
  for (unsigned ii = 0; ii < OPT->LCs(_prn.system()).size(); ii++) {
    t_lc::type tLC = OPT->LCs(_prn.system())[ii];
    if (!isValid(tLC)) {
      _valid = false;
      return;
    }
  }

  // Find Glonass Channel Number
  // ---------------------------
  if (_prn.system() == 'R') {
    _channel = PPP_CLIENT->ephPool()->getChannel(_prn);
  }
  else {
    _channel = 0;
  }

  // Compute Satellite Coordinates at Time of Transmission
  // -----------------------------------------------------
  _xcSat.ReSize(4); _xcSat = 0.0;
  _vvSat.ReSize(4); _vvSat = 0.0;
  bool totOK  = false;
  ColumnVector satPosOld(4); satPosOld = 0.0;
  t_lc::type tLC = isValid(t_lc::cIF) ? t_lc::cIF : t_lc::c1;
  double prange = obsValue(tLC);
  for (int ii = 1; ii <= 10; ii++) {
    bncTime ToT = _time - prange / t_CST::c - _xcSat[3];
    if (PPP_CLIENT->ephPool()->getCrd(_prn, ToT, _xcSat, _vvSat) != success) {
      _valid = false;
      return;
    }
    ColumnVector dx = _xcSat - satPosOld;
    dx[3] *= t_CST::c;
    if (dx.norm_Frobenius() < 1.e-4) {
      totOK = true;
      break;
    }
    satPosOld = _xcSat; 
  }
  if (totOK) {
    _model._satClkM = _xcSat[3] * t_CST::c; 
  }
  else {
    _valid = false;
  }
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppSatObs::lcCoeff(t_lc::type tLC, 
                          map<t_frequency::type, double>& codeCoeff,
                          map<t_frequency::type, double>& phaseCoeff) const {

  codeCoeff.clear();
  phaseCoeff.clear();

  double f1 = t_CST::freq(_fType1, _channel);
  double f2 = t_CST::freq(_fType2, _channel);

  switch (tLC) {
  case t_lc::l1:
    phaseCoeff[_fType1] = 1.0;  
    return;
  case t_lc::l2:  
    phaseCoeff[_fType2] = 1.0;  
    return;
  case t_lc::lIF: 
    phaseCoeff[_fType1] =  f1 * f1 / (f1 * f1 - f2 * f2);
    phaseCoeff[_fType2] = -f2 * f2 / (f1 * f1 - f2 * f2);
    return;
  case t_lc::MW:  
    phaseCoeff[_fType1] =  f1 / (f1 - f2);
    phaseCoeff[_fType2] = -f2 / (f1 - f2);
    codeCoeff[_fType1]  = -f1 / (f1 + f2);
    codeCoeff[_fType2]  = -f2 / (f1 + f2);
    return;
  case t_lc::CL:  
    phaseCoeff[_fType1] =  0.5;
    codeCoeff[_fType1]  =  0.5;
    return;
  case t_lc::c1:  
    codeCoeff[_fType1] = 1.0;  
    return;
  case t_lc::c2:  
    codeCoeff[_fType2] = 1.0;  
    return;
  case t_lc::cIF: 
    codeCoeff[_fType1] =  f1 * f1 / (f1 * f1 - f2 * f2);
    codeCoeff[_fType2] = -f2 * f2 / (f1 * f1 - f2 * f2);
    return;
  case t_lc::dummy: 
  case t_lc::maxLc: 
    return;
  }
}

// 
////////////////////////////////////////////////////////////////////////////
bool t_pppSatObs::isValid(t_lc::type tLC) const {
  bool valid = true;
  obsValue(tLC, &valid);
  return valid;
}
// 
////////////////////////////////////////////////////////////////////////////
double t_pppSatObs::obsValue(t_lc::type tLC, bool* valid) const {

  map<t_frequency::type, double> codeCoeff;
  map<t_frequency::type, double> phaseCoeff;
  lcCoeff(tLC, codeCoeff, phaseCoeff);

  double retVal = 0.0;
  if (valid) *valid = true;

  map<t_frequency::type, double>::const_iterator it;
  for (it = codeCoeff.begin(); it != codeCoeff.end(); it++) {
    t_frequency::type tFreq = it->first;
    if (_obs[tFreq] == 0) {
      if (valid) *valid = false;
      return 0.0;
    }
    else {
      retVal += it->second * _obs[tFreq]->_code;
    }
  }
  for (it = phaseCoeff.begin(); it != phaseCoeff.end(); it++) {
    t_frequency::type tFreq = it->first;
    if (_obs[tFreq] == 0) {
      if (valid) *valid = false;
      return 0.0;
    }
    else {
      retVal += it->second * _obs[tFreq]->_phase * t_CST::lambda(tFreq, _channel);
    }
  }

  return retVal;
}

// 
////////////////////////////////////////////////////////////////////////////
double t_pppSatObs::lambda(t_lc::type tLC) const {

  double f1 = t_CST::freq(_fType1, _channel);
  double f2 = t_CST::freq(_fType2, _channel);

  if      (tLC == t_lc::l1) {
    return t_CST::c / f1;
  }
  else if (tLC == t_lc::l2) {
    return t_CST::c / f2;
  }
  else if (tLC == t_lc::lIF) {
    return t_CST::c / (f1 + f2);
  }
  else if (tLC == t_lc::MW) {
    return t_CST::c / (f1 - f2);
  }
  else if (tLC == t_lc::CL) {
    return t_CST::c / f1 / 2.0;
  }

  return 0.0;
}

// 
////////////////////////////////////////////////////////////////////////////
double t_pppSatObs::sigma(t_lc::type tLC) const {

  map<t_frequency::type, double> codeCoeff;
  map<t_frequency::type, double> phaseCoeff;
  lcCoeff(tLC, codeCoeff, phaseCoeff);

  double retVal = 0.0;

  map<t_frequency::type, double>::const_iterator it;
  for (it = codeCoeff.begin(); it != codeCoeff.end(); it++) {
    retVal += it->second * it->second * OPT->_sigmaC1 * OPT->_sigmaC1;
  }
  for (it = phaseCoeff.begin(); it != phaseCoeff.end(); it++) {
    retVal += it->second * it->second * OPT->_sigmaL1 * OPT->_sigmaL1;
  }
  
  retVal = sqrt(retVal);    

  // Elevation-Dependent Weighting
  // -----------------------------
  double cEle = 1.0;
  if ( (OPT->_eleWgtCode  && t_lc::includesCode(tLC)) ||
       (OPT->_eleWgtPhase && t_lc::includesPhase(tLC)) ) {
    double eleD = eleSat()*180.0/M_PI;
    double hlp  = fabs(90.0 - eleD);
    cEle = (1.0 + hlp*hlp*hlp*0.000004);
  }

  return cEle * retVal;
}

// 
////////////////////////////////////////////////////////////////////////////
double t_pppSatObs::maxRes(t_lc::type tLC) const {

  map<t_frequency::type, double> codeCoeff;
  map<t_frequency::type, double> phaseCoeff;
  lcCoeff(tLC, codeCoeff, phaseCoeff);

  double retVal = 0.0;

  map<t_frequency::type, double>::const_iterator it;
  for (it = codeCoeff.begin(); it != codeCoeff.end(); it++) {
    retVal += it->second * OPT->_maxResC1;
  }
  for (it = phaseCoeff.begin(); it != phaseCoeff.end(); it++) {
    retVal += it->second * OPT->_maxResL1;
  }

  return retVal;
}


// 
////////////////////////////////////////////////////////////////////////////
t_irc t_pppSatObs::cmpModel(const t_pppStation* station) {

  // Reset all model values
  // ----------------------
  _model.reset();

  // Topocentric Satellite Position
  // ------------------------------
  ColumnVector rSat = _xcSat.Rows(1,3);
  ColumnVector rhoV = rSat - station->xyzApr(); 
  _model._rho = rhoV.norm_Frobenius();

  ColumnVector neu(3);
  xyz2neu(station->ellApr().data(), rhoV.data(), neu.data());

  _model._eleSat = acos( sqrt(neu[0]*neu[0] + neu[1]*neu[1]) / _model._rho );
  if (neu[2] < 0) {
    _model._eleSat *= -1.0;
  }
  _model._azSat  = atan2(neu[1], neu[0]);

  // Satellite Clocks
  // ----------------
  _model._satClkM = _xcSat[3] * t_CST::c;

  // Receiver Clocks
  // ---------------
  _model._recClkM = station->dClk() * t_CST::c;

  // Sagnac Effect (correction due to Earth rotation)
  // ------------------------------------------------
  ColumnVector Omega(3);
  Omega[0] = 0.0;
  Omega[1] = 0.0;
  Omega[2] = t_CST::omega / t_CST::c;
  _model._sagnac = DotProduct(Omega, crossproduct(rSat, station->xyzApr()));

  // Antenna Eccentricity
  // --------------------
  _model._antEcc = -DotProduct(station->xyzEcc(), rhoV) / _model._rho;

  // Antenna Phase Center Offsets and Variations
  // -------------------------------------------
  if (PPP_CLIENT->antex()) {
    bool found;
    double pco = PPP_CLIENT->antex()->rcvCorr(station->antName(), _model._eleSat, found);
    for (unsigned ii = 0; ii < t_frequency::max; ii++) {
      _model._antPCO[ii] = pco;
    }
  }

  // Tropospheric Delay
  // ------------------
  _model._tropo = t_tropo::delay_saast(station->xyzApr(), _model._eleSat);

  // Phase Wind-Up
  // -------------
  _model._windUp = station->windUp(_time, _prn, rSat);

  // Code and Phase Biases
  // ---------------------
  const t_satBias* satBias = PPP_CLIENT->obsPool()->satBias(_prn);
  if (satBias) { 
    for (unsigned ii = 0; ii < satBias->_bias.size(); ii++) {
      const t_frqBias& bias = satBias->_bias[ii];
      for (unsigned iFreq = 1; iFreq < t_frequency::max; iFreq++) {
        const t_frqObs* obs = _obs[iFreq];
        if (obs && obs->_rnxType2ch == bias._rnxType2ch) {
          _model._codeBias[iFreq]  = bias._code;
          _model._phaseBias[iFreq] = bias._phase;
        }
      }
    }
  }

  // Tidal Correction
  // ----------------
  _model._tide = -DotProduct(station->tideDspl(), rhoV) / _model._rho;

  // Ionospheric Delay
  // -----------------
  // TODO

  // Ocean Loading
  // -------------
  // TODO

  // Set Model Set Flag
  // ------------------
  _model._set = true;

  return success;
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppSatObs::printModel() const {
  LOG.setf(ios::fixed);
  LOG << "MODEL for Satellite " << _prn.toString() << endl
      << "RHO:        " << setw(12) << setprecision(3) << _model._rho     << endl
      << "ELE:        " << setw(12) << setprecision(3) << _model._eleSat * 180.0 / M_PI << endl
      << "AZI:        " << setw(12) << setprecision(3) << _model._azSat  * 180.0 / M_PI << endl
      << "SATCLK:     " << setw(12) << setprecision(3) << _model._satClkM << endl
      << "RECCLK:     " << setw(12) << setprecision(3) << _model._recClkM << endl
      << "SAGNAC:     " << setw(12) << setprecision(3) << _model._sagnac  << endl
      << "ANTECC:     " << setw(12) << setprecision(3) << _model._antEcc  << endl
      << "TROPO:      " << setw(12) << setprecision(3) << _model._tropo   << endl
      << "WINDUP:     " << setw(12) << setprecision(3) << _model._windUp  << endl
      << "TIDES:      " << setw(12) << setprecision(3) << _model._tide    << endl;
  for (unsigned iFreq = 1; iFreq < t_frequency::max; iFreq++) {
    if (_obs[iFreq]) {
      LOG << "PCO:        " << t_frequency::toString(t_frequency::type(iFreq)) << setw(12) << setprecision(3) << _model._antPCO[iFreq]    << endl
          << "BIAS CODE:  " << t_frequency::toString(t_frequency::type(iFreq)) << setw(12) << setprecision(3) << _model._codeBias[iFreq]  << endl
          << "BIAS PHASE: " << t_frequency::toString(t_frequency::type(iFreq)) << setw(12) << setprecision(3) << _model._phaseBias[iFreq] << endl;
    }
  }
  LOG << "OBS-CMP P3: " << _prn.toString() << " " 
      << setw(12) << setprecision(3) << obsValue(t_lc::cIF) << " "
      << setw(12) << setprecision(3) << cmpValue(t_lc::cIF) << " "
      << setw(12) << setprecision(3) << obsValue(t_lc::cIF) - cmpValue(t_lc::cIF) << endl;

  LOG << "OBS-CMP L3: " << _prn.toString() << " " 
      << setw(12) << setprecision(3) << obsValue(t_lc::lIF) << " "
      << setw(12) << setprecision(3) << cmpValue(t_lc::lIF) << " "
      << setw(12) << setprecision(3) << obsValue(t_lc::lIF) - cmpValue(t_lc::lIF) << endl;

  LOG << "OBS-CMP MW: " << _prn.toString() << " " 
      << setw(12) << setprecision(3) << obsValue(t_lc::MW) << " "
      << setw(12) << setprecision(3) << cmpValue(t_lc::MW) << " "
      << setw(12) << setprecision(3) << obsValue(t_lc::MW) - cmpValue(t_lc::MW) << endl;
}

// 
////////////////////////////////////////////////////////////////////////////
double t_pppSatObs::cmpValueForBanc(t_lc::type tLC) const {
  return cmpValue(tLC) - _model._rho - _model._sagnac - _model._recClkM;
}

// 
////////////////////////////////////////////////////////////////////////////
double t_pppSatObs::cmpValue(t_lc::type tLC) const {

  if (!isValid(tLC)) {
    return 0.0;
  }

  // Non-Dispersive Part
  // -------------------
  double nonDisp = _model._rho    + _model._recClkM - _model._satClkM 
                 + _model._sagnac + _model._antEcc  + _model._tropo 
                 + _model._tide;

  // Add Dispersive Part
  // -------------------
  map<t_frequency::type, double> codeCoeff;
  map<t_frequency::type, double> phaseCoeff;
  lcCoeff(tLC, codeCoeff, phaseCoeff);

  double dispPart = 0.0;

  map<t_frequency::type, double>::const_iterator it;
  for (it = codeCoeff.begin(); it != codeCoeff.end(); it++) {
    t_frequency::type tFreq = it->first;
    dispPart += it->second * (_model._antPCO[tFreq] + _model._codeBias[tFreq]);
  }
  for (it = phaseCoeff.begin(); it != phaseCoeff.end(); it++) {
    t_frequency::type tFreq = it->first;
    dispPart += it->second * (_model._antPCO[tFreq] + _model._phaseBias[tFreq]);
  }

//  double L1 = nonDisp + _model._antPco1 - _model._biasL1 + _model._windUp * t_CST::c / _f1;
//  double L2 = nonDisp + _model._antPco2 - _model._biasL2 + _model._windUp * t_CST::c / _f2;
//  double C1 = nonDisp + _model._antPco1 - _model._biasC1;
//  double C2 = nonDisp + _model._antPco2 - _model._biasC2;

    return nonDisp + dispPart;
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppSatObs::setRes(t_lc::type tLC, double res) {
  _res[tLC] = res;
}

// 
////////////////////////////////////////////////////////////////////////////
double t_pppSatObs::getRes(t_lc::type tLC) const {
  map<t_lc::type, double>::const_iterator it = _res.find(tLC);
  if (it != _res.end()) {
    return it->second;
  }
  else {
    return 0.0;
  }
}
