
// Part of BNC, a utility for retrieving decoding and
// converting GNSS data streams from NTRIP broadcasters.
//
// Copyright (C) 2007
// German Federal Agency for Cartography and Geodesy (BKG)
// http://www.bkg.bund.de
// Czech Technical University Prague, Department of Geodesy
// http://www.fsv.cvut.cz
//
// Email: euref-ip@bkg.bund.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation, version 2.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

/* -------------------------------------------------------------------------
 * BKG NTRIP Client
 * -------------------------------------------------------------------------
 *
 * Class:      t_pppThread, t_pppRun
 *
 * Purpose:    Single PPP Client (running in its own thread)
 *
 * Author:     L. Mervart
 *
 * Created:    29-Jul-2014
 *
 * Changes:    
 *
 * -----------------------------------------------------------------------*/


#include <iostream>
#include <iomanip>
#include <string.h>
#include <map>

#include "pppThread.h"
#include "bnccore.h"

using namespace BNC_PPP;
using namespace std;

// Constructor
////////////////////////////////////////////////////////////////////////////
t_pppThread::t_pppThread(const t_pppOptions* opt) : QThread(0) {
  _opt   = opt;
  _pppRun = 0;
  connect(this, SIGNAL(finished()), this, SLOT(deleteLater()));

  connect(this, SIGNAL(newMessage(QByteArray,bool)), 
          BNC_CORE, SLOT(slotMessage(const QByteArray,bool)));
}

// Destructor
////////////////////////////////////////////////////////////////////////////
t_pppThread::~t_pppThread() {
  delete _pppRun;
}

// Run (virtual)
////////////////////////////////////////////////////////////////////////////
void t_pppThread::run() {
  try {
    _pppRun = new t_pppRun(_opt);
    QThread::exec();
  }
  catch (pppExcept exc) {
    _pppRun = 0;
    emit newMessage(QByteArray(exc.what().c_str()), true);
  }
}

// Constructor
////////////////////////////////////////////////////////////////////////////
t_pppRun::t_pppRun(const t_pppOptions* opt) {
  _opt = opt;
  connect(this, SIGNAL(newMessage(QByteArray,bool)), 
          BNC_CORE, SLOT(slotMessage(const QByteArray,bool)));
  if (_opt->_realTime) {
    connect(BNC_CORE->caster(), SIGNAL(newObs(QByteArray, QList<t_obs>)),
            this, SLOT(slotNewObs(QByteArray, QList<t_obs>)));

    connect(BNC_CORE, SIGNAL(newEphGPS(gpsephemeris)),
            this, SLOT(slotNewEphGPS(gpsephemeris)));
  
    connect(BNC_CORE, SIGNAL(newEphGlonass(glonassephemeris)),
            this, SLOT(slotNewEphGlonass(glonassephemeris)));
  
    connect(BNC_CORE, SIGNAL(newEphGalileo(galileoephemeris)),
            this, SLOT(slotNewEphGalileo(galileoephemeris)));

    connect(BNC_CORE, SIGNAL(newCorrections(QStringList)),
            this, SLOT(slotNewCorrections(QStringList)));

    _pppClient = new t_pppClient(_opt);
  }
  else {
    throw pppExcept("t_pppRun: post-processing not yet implemented");
  }
}

// Destructor
////////////////////////////////////////////////////////////////////////////
t_pppRun::~t_pppRun() {
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewEphGPS(gpsephemeris gpseph) {
  QMutexLocker locker(&_mutex);
  t_ephGPS eph;
  eph.set(&gpseph);
  _pppClient->putEphemeris(&eph);
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewEphGlonass(glonassephemeris gloeph) {
  QMutexLocker locker(&_mutex);
  t_ephGlo eph;
  eph.set(&gloeph);
  _pppClient->putEphemeris(&eph);
}
  
// 
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewEphGalileo(galileoephemeris galeph) {
  QMutexLocker locker(&_mutex);
  t_ephGal eph;
  eph.set(&galeph);
  _pppClient->putEphemeris(&eph);
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewCorrections(QStringList corrList) {
  QMutexLocker locker(&_mutex);

  QStringListIterator it(corrList);
  while (it.hasNext()) {
    it.next();
  }

  // _pppClient->putOrbCorrections(const std::vector<t_orbCorr*>& corr); 
  // _pppClient->putClkCorrections(const std::vector<t_clkCorr*>& corr); 
  // _pppClient->putBiases(const std::vector<t_satBiases*>& biases);   
}
//
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewObs(QByteArray staID, QList<t_obs> obsList) {
  QMutexLocker locker(&_mutex);

  if (string(staID.data()) != _opt->_roverName) {
    return;
  }

  // Loop over all obsevations (possible different epochs)
  // -----------------------------------------------------
  QListIterator<t_obs> it(obsList);
  while (it.hasNext()) {
    const t_obs& oldObs = it.next();
    t_satObs* newObs = new t_satObs;
   
    newObs->_prn.set(oldObs.satSys, oldObs.satNum);
    newObs->_time.set(oldObs.GPSWeek, oldObs.GPSWeeks);

    // Find the corresponding data epoch or create a new one
    // -----------------------------------------------------
    t_pppEpoData* epoData = 0;
    deque<t_pppEpoData*>::const_iterator it;
    for (it = _pppEpochs.begin(); it != _pppEpochs.end(); it++) {
      if (newObs->_time == (*it)->_time) {
        epoData = *it;
        break;
      }
    }
    if (epoData == 0) {
      if (_pppEpochs.empty() || newObs->_time > _pppEpochs.back()->_time) {
        epoData = new t_pppEpoData;
        epoData->_time = newObs->_time;
        _pppEpochs.push_back(epoData);
      }
    }

    // Fill the new observation and add it to the corresponding epoch
    // --------------------------------------------------------------
    if (epoData != 0) {
      epoData->_satObs.push_back(newObs);
      map<string, t_frqObs*> pppObsMap;
      for (unsigned iEntry = 0; iEntry < GNSSENTRY_NUMBER; iEntry++) {
        string hlp(oldObs.rnxStr(iEntry).toAscii().data());
        if (hlp.length() == 3) {
          char   obsType    = hlp[0];
          string rnxType2ch = hlp.substr(1);
          if (obsType == 'C' || obsType == 'L') {
            t_frqObs* pppObs = 0;
            if (pppObsMap.find(rnxType2ch) == pppObsMap.end()) {
              pppObs                = new t_frqObs();
              pppObsMap[rnxType2ch] = pppObs;
              pppObs->_rnxType2ch   = rnxType2ch;
              newObs->_obs.push_back(pppObs);
            }
            else {
              pppObs = pppObsMap[rnxType2ch];
            }
            if      (obsType == 'C') {
              pppObs->_code      = oldObs._measdata[iEntry];
              pppObs->_codeValid = true;
            }
            else if (obsType == 'L') {
              pppObs->_phase      = oldObs._measdata[iEntry];
              pppObs->_phaseValid = true;
            }
          }
        }
      }
    }
  }

  // Process the oldest epoch
  // ------------------------
  if (_pppEpochs.size() > 1) {

    const vector<t_satObs*>& satObs = _pppEpochs.front()->_satObs;

    t_output output;
    _pppClient->processEpoch(satObs, &output);

    delete _pppEpochs.front(); _pppEpochs.pop_front();
    
    emit newMessage(QByteArray(output._log.c_str()), true);
  }
}
    
