
// Part of BNC, a utility for retrieving decoding and
// converting GNSS data streams from NTRIP broadcasters.
//
// Copyright (C) 2007
// German Federal Agency for Cartography and Geodesy (BKG)
// http://www.bkg.bund.de
// Czech Technical University Prague, Department of Geodesy
// http://www.fsv.cvut.cz
//
// Email: euref-ip@bkg.bund.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation, version 2.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

/* -------------------------------------------------------------------------
 * BKG NTRIP Client
 * -------------------------------------------------------------------------
 *
 * Class:      t_pppClient
 *
 * Purpose:    PPP Client processing starts here
 *
 * Author:     L. Mervart
 *
 * Created:    29-Jul-2014
 *
 * Changes:    
 *
 * -----------------------------------------------------------------------*/

#include <QThreadStorage>

#include <iostream>
#include <iomanip>
#include <stdlib.h>
#include <string.h>
#include <stdexcept>

#include "pppClient.h"
#include "ephpool.h"
#include "obspool.h"
#include "satbias.h"
#include "bncconst.h"
#include "bncutils.h"
#include "station.h"
#include "bncantex.h"
#include "filter.h"

using namespace BNC;
using namespace std;

// Global variable holding thread-specific pointers
//////////////////////////////////////////////////////////////////////////////
QThreadStorage<t_pppClient*> CLIENTS;

// Static function returning thread-specific pointer
//////////////////////////////////////////////////////////////////////////////
t_pppClient* t_pppClient::instance() {
  return CLIENTS.localData();
}

// Constructor
//////////////////////////////////////////////////////////////////////////////
t_pppClient::t_pppClient(const t_options* opt) {
  _output   = 0;
  _opt      = new t_options(*opt);
  _log      = new ostringstream();
  _ephPool  = new t_ephPool();
  _obsPool  = new t_obsPool();
  _staRover = new t_station();
  _filter   = new t_filter();
  _tides    = new t_tides();

  if (!_opt->_antexFile.empty()) {
    _antex = new bncAntex(_opt->_antexFile.c_str());
  }
  else {
    _antex = 0;
  }

  CLIENTS.setLocalData(this);  // CLIENTS takes ownership over "this"
}

// Destructor
//////////////////////////////////////////////////////////////////////////////
t_pppClient::~t_pppClient() {
  delete _log;
  delete _opt;
  delete _ephPool;
  delete _obsPool;
  delete _staRover;
  delete _antex;
  delete _filter;
  delete _tides;
  clearObs();
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppClient::putEphemeris(const t_eph* eph) {
  const t_ephGPS* ephGPS = dynamic_cast<const t_ephGPS*>(eph);
  const t_ephGlo* ephGlo = dynamic_cast<const t_ephGlo*>(eph);
  const t_ephGal* ephGal = dynamic_cast<const t_ephGal*>(eph);
  if      (ephGPS) {
    _ephPool->putEphemeris(new t_ephGPS(*ephGPS));
  }
  else if (ephGlo) {
    _ephPool->putEphemeris(new t_ephGlo(*ephGlo));
  }
  else if (ephGal) {
    _ephPool->putEphemeris(new t_ephGal(*ephGal));
  }
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppClient::putOrbCorrections(const vector<t_orbCorr*>& corr) {
  for (unsigned ii = 0; ii < corr.size(); ii++) {
    _ephPool->putOrbCorrection(new t_orbCorr(*corr[ii]));
  }
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppClient::putClkCorrections(const vector<t_clkCorr*>& corr) {
  for (unsigned ii = 0; ii < corr.size(); ii++) {
    _ephPool->putClkCorrection(new t_clkCorr(*corr[ii]));
  }
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppClient::putBiases(const vector<t_satBiases*>& biases) {
  for (unsigned ii = 0; ii < biases.size(); ii++) {
    _obsPool->putBiases(new t_satBias(*biases[ii]));
  }
}

// 
//////////////////////////////////////////////////////////////////////////////
t_irc t_pppClient::prepareObs(const vector<t_pppSatObs*>& pppSatObs,
                              vector<t_satObs*>& obsVector, bncTime& epoTime) {
  // Default 
  // -------
  epoTime.reset();

  // Create vector of valid observations
  // -----------------------------------
  int numValidGPS = 0;
  for (unsigned ii = 0; ii < pppSatObs.size(); ii++) {
    char system = pppSatObs[ii]->_prn.system();
    if (system == 'G' || (system == 'R' && OPT->useGlonass())) {
      t_satObs* satObs = new t_satObs(*pppSatObs[ii]);
      if (satObs->isValid()) {
        obsVector.push_back(satObs);
        if (satObs->prn().system() == 'G') {
          ++numValidGPS;
        }
      }
      else {
        delete satObs;
      }
    }
  }

  // Check whether data are synchronized, compute epoTime
  // ----------------------------------------------------
  const double MAXSYNC = 0.05; // synchronization limit
  double meanDt = 0.0;
  for (unsigned ii = 0; ii < obsVector.size(); ii++) {
    const t_satObs* satObs = obsVector.at(ii);
    if (epoTime.undef()) {
      epoTime = satObs->time();
    }
    else {
      double dt = satObs->time() - epoTime;
      if (fabs(dt) > MAXSYNC) {
        LOG << "t_pppClient::prepareObs asynchronous observations" << endl;
        return failure;
      }
      meanDt += dt;
    }
  }

  if (obsVector.size() > 0) {
    epoTime += meanDt / obsVector.size();
  }

  return success;
}

// Compute the Bancroft position, check for blunders
//////////////////////////////////////////////////////////////////////////////
t_irc t_pppClient::cmpBancroft(const bncTime& epoTime, 
                                  vector<t_satObs*>& obsVector,
                                  ColumnVector& xyzc, bool print) {

  t_lc::type tLC = (OPT->dualFreqRequired() ? t_lc::cIF : t_lc::c1);

  while (true) {
    Matrix BB(obsVector.size(), 4);
    int iObs = -1;
    for (unsigned ii = 0; ii < obsVector.size(); ii++) {
      const t_satObs* satObs = obsVector.at(ii);
      if ( satObs->isValid() && satObs->prn().system() == 'G' &&
           (!satObs->modelSet() || satObs->eleSat() >= OPT->_minEle) ) {
        ++iObs;   
        BB[iObs][0] = satObs->xc()[0];
        BB[iObs][1] = satObs->xc()[1];
        BB[iObs][2] = satObs->xc()[2];
        BB[iObs][3] = satObs->obsValue(tLC) - satObs->cmpValueForBanc(tLC);
      }
    }
    if (iObs + 1 < OPT->_minObs) {
      LOG << "t_pppClient::cmpBancroft not enough observations" << endl;
      return failure;
    }
    BB = BB.Rows(1,iObs+1);
    bancroft(BB, xyzc);

    xyzc[3] /= t_CST::c;

    // Check Blunders
    // --------------
    const double BLUNDER = 100.0;
    double   maxRes      = 0.0;
    unsigned maxResIndex = 0;
    for (unsigned ii = 0; ii < obsVector.size(); ii++) {
      const t_satObs* satObs = obsVector.at(ii);
      if ( satObs->isValid() && satObs->prn().system() == 'G' &&
           (!satObs->modelSet() || satObs->eleSat() >= OPT->_minEle) ) {
        ColumnVector rr = satObs->xc().Rows(1,3) - xyzc.Rows(1,3);
        double res = rr.norm_Frobenius() - satObs->obsValue(tLC) 
          - (satObs->xc()[3] - xyzc[3]) * t_CST::c;
        if (fabs(res) > maxRes) {
          maxRes      = fabs(res);
          maxResIndex = ii;
        }
      }
    }
    if (maxRes < BLUNDER) {
      if (print) {
        LOG.setf(ios::fixed);
        LOG << string(epoTime) << " BANCROFT:"        << ' '
            << setw(14) << setprecision(3) << xyzc[0] << ' '
            << setw(14) << setprecision(3) << xyzc[1] << ' '
            << setw(14) << setprecision(3) << xyzc[2] << ' '
            << setw(14) << setprecision(3) << xyzc[3] * t_CST::c << endl << endl;
      }
      break;
    }
    else {
      t_satObs* satObs = obsVector.at(maxResIndex);
      LOG << "t_pppClient::cmpBancroft outlier " << satObs->prn().toString()
          << " " << maxRes << endl;
      delete satObs;
      obsVector.erase(obsVector.begin() + maxResIndex);
    }
  }

  return success;
}

// Compute A Priori GPS-Glonass Offset
//////////////////////////////////////////////////////////////////////////////
double t_pppClient::cmpOffGG(vector<t_satObs*>& obsVector) {

  t_lc::type tLC   = (OPT->dualFreqRequired() ? t_lc::cIF : t_lc::c1);
  double     offGG = 0.0;

  if (OPT->useGlonass()) {
    while (true) {
      offGG = 0.0;
      bool outlierFound = false;
      unsigned nObs  = 0;
      for (unsigned ii = 0; ii < obsVector.size(); ii++) {
        t_satObs* satObs = obsVector.at(ii);
        if ( !satObs->outlier() && satObs->isValid() && satObs->prn().system() == 'R' &&
             (!satObs->modelSet() || satObs->eleSat() >= OPT->_minEle) ) {
          ++nObs;
          double ll = satObs->obsValue(tLC) - satObs->cmpValue(tLC);
          if (fabs(ll) > 1000.0) {
            satObs->setOutlier();
            outlierFound = true;
           LOG << "t_pppClient::cmpOffGG outlier " << satObs->prn().toString()
               << " " << ll << endl;
          }
          offGG += ll;
        }
      }
      if (nObs > 0) {
        offGG = offGG / nObs;
      }
      else {
        offGG = 0.0;
      }
      if (!outlierFound) {
        break;
      }
    }
  }

  return offGG;
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppClient::initOutput(t_output* output) {
  _output = output;
  _output->_numSat = 0;
  _output->_pDop   = 0.0;
  _output->_error  = false;
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppClient::clearObs() {
  for (unsigned ii = 0; ii < _obsRover.size(); ii++) {
    delete _obsRover.at(ii);
  }
  _obsRover.clear();
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppClient::finish(t_irc irc) {

  clearObs();

  _output->_epoTime = _epoTimeRover;

  if (irc == success) {
    _output->_xyzRover[0] = _staRover->xyzApr()[0] + _filter->x()[0];
    _output->_xyzRover[1] = _staRover->xyzApr()[1] + _filter->x()[1];
    _output->_xyzRover[2] = _staRover->xyzApr()[2] + _filter->x()[2];
    copy(&_filter->Q().data()[0], &_filter->Q().data()[6], _output->_covMatrix);
    _output->_numSat     = _filter->numSat();
    _output->_pDop       = _filter->PDOP();
    _output->_error = false;
  }
  else {
    _output->_error = true;
  }
  _output->_log = _log->str();
  delete _log; _log = new ostringstream();
}

// 
//////////////////////////////////////////////////////////////////////////////
t_irc t_pppClient::cmpModel(t_station* station, const ColumnVector& xyzc,
                               vector<t_satObs*>& obsVector) {

  bncTime time;
  time = _epoTimeRover;
  station->setName(OPT->_roverName);
  station->setAntName(OPT->_antNameRover);
  if (OPT->xyzAprRoverSet()) {
    station->setXyzApr(OPT->_xyzAprRover);
  }
  else {
    station->setXyzApr(xyzc.Rows(1,3));
  }
  station->setNeuEcc(OPT->_neuEccRover);

  // Receiver Clock
  // --------------
  station->setDClk(xyzc[3]);

  // Tides
  // -----
  station->setTideDspl( _tides->displacement(time, station->xyzApr()) );
  
  // Observation model
  // -----------------
  vector<t_satObs*>::iterator it = obsVector.begin();
  while (it != obsVector.end()) {
    t_satObs* satObs = *it;
    satObs->cmpModel(station);
    if (satObs->isValid() && satObs->eleSat() >= OPT->_minEle) {
      ++it;
    }
    else {
      delete satObs;
      it = obsVector.erase(it);
    }
  }

  return success;
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppClient::processEpoch(const vector<t_pppSatObs*>& pppSatObs, t_output* output) {

  try {
    initOutput(output);

    // Prepare Observations of the Rover
    // ---------------------------------    
    if (prepareObs(pppSatObs, _obsRover, _epoTimeRover) != success) {
      return finish(failure);
    }

    LOG << "\nResults of Epoch ";
    if (!_epoTimeRover.undef()) LOG << string(_epoTimeRover);
    LOG << "\n--------------------------------------\n";
 
    for (int iter = 1; iter <= 2; iter++) {
      ColumnVector xyzc(4); xyzc = 0.0;
      bool print = (iter == 2);
      if (cmpBancroft(_epoTimeRover, _obsRover, xyzc, print) != success) {
        return finish(failure);
      }
      if (cmpModel(_staRover, xyzc, _obsRover) != success) {
        return finish(failure);
      }
    }

    _offGG = cmpOffGG(_obsRover);

    // Store last epoch of data
    // ------------------------    
    _obsPool->putEpoch(_epoTimeRover, _obsRover);

    // Process Epoch in Filter
    // -----------------------
    if (_filter->processEpoch(_obsPool) != success) {
      return finish(failure);
    }
  }
  catch (Exception& exc) {
    LOG << exc.what() << endl;
    return finish(failure);
  }
  catch (pppExcept msg) {
    LOG << msg.what() << endl;
    return finish(failure);
  }
  catch (const char* msg) {
    LOG << msg << endl;
    return finish(failure);
  }
  catch (...) {
    LOG << "unknown exception" << endl;
    return finish(failure);
  }

  return finish(success);
}

// 
////////////////////////////////////////////////////////////////////////////
double lorentz(const ColumnVector& aa, const ColumnVector& bb) {
  return aa[0]*bb[0] +  aa[1]*bb[1] +  aa[2]*bb[2] -  aa[3]*bb[3];
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppClient::bancroft(const Matrix& BBpass, ColumnVector& pos) {

  if (pos.Nrows() != 4) {
    pos.ReSize(4);
  }
  pos = 0.0;

  for (int iter = 1; iter <= 2; iter++) {
    Matrix BB = BBpass;
    int mm = BB.Nrows();
    for (int ii = 1; ii <= mm; ii++) {
      double xx = BB(ii,1);
      double yy = BB(ii,2);
      double traveltime = 0.072;
      if (iter > 1) {
        double zz  = BB(ii,3);
        double rho = sqrt( (xx-pos(1)) * (xx-pos(1)) + 
                           (yy-pos(2)) * (yy-pos(2)) + 
                           (zz-pos(3)) * (zz-pos(3)) );
        traveltime = rho / t_CST::c;
      }
      double angle = traveltime * t_CST::omega;
      double cosa  = cos(angle);
      double sina  = sin(angle);
      BB(ii,1) =  cosa * xx + sina * yy;
      BB(ii,2) = -sina * xx + cosa * yy;
    }
    
    Matrix BBB;
    if (mm > 4) {
      SymmetricMatrix hlp; hlp << BB.t() * BB;
      BBB = hlp.i() * BB.t();
    }
    else {
      BBB = BB.i();
    }
    ColumnVector ee(mm); ee = 1.0;
    ColumnVector alpha(mm); alpha = 0.0;
    for (int ii = 1; ii <= mm; ii++) {
      alpha(ii) = lorentz(BB.Row(ii).t(),BB.Row(ii).t())/2.0; 
    }
    ColumnVector BBBe     = BBB * ee;
    ColumnVector BBBalpha = BBB * alpha;
    double aa = lorentz(BBBe, BBBe);
    double bb = lorentz(BBBe, BBBalpha)-1;
    double cc = lorentz(BBBalpha, BBBalpha);
    double root = sqrt(bb*bb-aa*cc);

    Matrix hlpPos(4,2); 
    hlpPos.Column(1) = (-bb-root)/aa * BBBe + BBBalpha;
    hlpPos.Column(2) = (-bb+root)/aa * BBBe + BBBalpha;

    ColumnVector omc(2);
    for (int pp = 1; pp <= 2; pp++) {
      hlpPos(4,pp)      = -hlpPos(4,pp);
      omc(pp) = BB(1,4) - 
                sqrt( (BB(1,1)-hlpPos(1,pp)) * (BB(1,1)-hlpPos(1,pp)) +
                      (BB(1,2)-hlpPos(2,pp)) * (BB(1,2)-hlpPos(2,pp)) +
                      (BB(1,3)-hlpPos(3,pp)) * (BB(1,3)-hlpPos(3,pp)) ) - 
                hlpPos(4,pp);
    }
    if ( fabs(omc(1)) > fabs(omc(2)) ) {
      pos = hlpPos.Column(2);
    }
    else {
      pos = hlpPos.Column(1);
    }
  }
}

