
// Part of BNC, a utility for retrieving decoding and
// converting GNSS data streams from NTRIP broadcasters.
//
// Copyright (C) 2007
// German Federal Agency for Cartography and Geodesy (BKG)
// http://www.bkg.bund.de
// Czech Technical University Prague, Department of Geodesy
// http://www.fsv.cvut.cz
//
// Email: euref-ip@bkg.bund.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation, version 2.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

/* -------------------------------------------------------------------------
 * BKG NTRIP Client
 * -------------------------------------------------------------------------
 *
 * Class:      t_pppMain
 *
 * Purpose:    Start of the PPP client(s)
 *
 * Author:     L. Mervart
 *
 * Created:    29-Jul-2014
 *
 * Changes:    
 *
 * -----------------------------------------------------------------------*/

#include <iostream>

#include "pppMain.h"
#include "bncsettings.h"

using namespace BNC;
using namespace std;

// Constructor
//////////////////////////////////////////////////////////////////////////////
t_pppMain::t_pppMain() {
  _running = false;
}

// Destructor
//////////////////////////////////////////////////////////////////////////////
t_pppMain::~t_pppMain() {
  stop();
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppMain::start() {
  if (_running) {
    return;
  }

  try {
    readOptions();

    QListIterator<t_options*> iOpt(_options);
    while (iOpt.hasNext()) {
      const t_options* opt = iOpt.next();
      t_pppThread* pppThread = new t_pppThread(opt);
      pppThread->start();
      _pppThreads << pppThread;
      _running = true;
    }
  }
  catch (pppExcept exc) {
    _running = true;
    stop();
  }
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppMain::stop() {
  if (!_running) {
    return;
  }

  QListIterator<t_pppThread*> it(_pppThreads);
  while (it.hasNext()) {
    t_pppThread* pppThread = it.next();
    pppThread->exit();
  }
  _pppThreads.clear();

  QListIterator<t_options*> iOpt(_options);
  while (iOpt.hasNext()) {
    delete iOpt.next();
  }
  _options.clear();

  _running = false;
}

// 
//////////////////////////////////////////////////////////////////////////////
void t_pppMain::readOptions() {

  QListIterator<t_options*> iOpt(_options);
  while (iOpt.hasNext()) {
    delete iOpt.next();
  }
  _options.clear();

  bncSettings settings;

  _logFile  = settings.value("PPP/logFile").toString();
  _nmeaFile = settings.value("PPP/nmeaFile").toString();
  _nmeaPort = settings.value("PPP/nmeaPort").toInt();

  bool realTime = false;
  if      (settings.value("PPP/dataSource").toString() == "Real-Time Streams") {
    realTime = true;
  }
  else if (settings.value("PPP/dataSource").toString() == "RINEX Files") {
    realTime = false;
  }
  else {
    return;    
  }

  QListIterator<QString> iSta(settings.value("PPP/staTable").toStringList());
  while (iSta.hasNext()) {
    QStringList hlp = iSta.next().split(",");

    if (hlp.size() < 9) {
      throw pppExcept("pppMain: wrong option staTable");
    }

    t_options* opt = new t_options();

    opt->_realTime    = realTime;
    opt->_roverName   = hlp[0].toAscii().data();
    opt->_sigCrd[0]   = hlp[1].toDouble();
    opt->_sigCrd[1]   = hlp[2].toDouble();
    opt->_sigCrd[2]   = hlp[3].toDouble();
    opt->_noiseCrd[0] = hlp[4].toDouble();
    opt->_noiseCrd[1] = hlp[5].toDouble();
    opt->_noiseCrd[2] = hlp[6].toDouble();
    opt->_sigTropo    = hlp[7].toDouble();
    opt->_noiseTropo  = hlp[8].toDouble();

    if (realTime) {
      opt->_corrMount = settings.value("PPP/corrMount").toString().toAscii().data();
    }
    else {
      opt->_rinexObs = settings.value("PPP/rinexObs").toString().toAscii().data();
      opt->_rinexNav = settings.value("PPP/rinexNav").toString().toAscii().data();
      opt->_corrFile = settings.value("PPP/corrFile").toString().toAscii().data();
    }

    opt->_crdFile   = settings.value("PPP/crdFile").toString().toAscii().data();
    opt->_antexFile = settings.value("PPP/antexFile").toString().toAscii().data();

    opt->_sigmaC1      = settings.value("PPP/sigmaC1").toDouble();
    opt->_sigmaL1      = settings.value("PPP/sigmaL1").toDouble();
    opt->_corrWaitTime = settings.value("PPP/corrWaitTime").toDouble();

    if      (settings.value("PPP/lcGPS").toString() == "P3") {
      opt->_lcGPS.push_back(t_lc::cIF);
    }
    else if (settings.value("PPP/lcGPS").toString() == "L3") {
      opt->_lcGPS.push_back(t_lc::lIF);
    }
    else if (settings.value("PPP/lcGPS").toString() == "P3&L3") {
      opt->_lcGPS.push_back(t_lc::cIF);
      opt->_lcGPS.push_back(t_lc::lIF);
    }

    if      (settings.value("PPP/lcGLONASS").toString() == "P3") {
      opt->_lcGLONASS.push_back(t_lc::cIF);
    }
    else if (settings.value("PPP/lcGLONASS").toString() == "L3") {
      opt->_lcGLONASS.push_back(t_lc::lIF);
    }
    else if (settings.value("PPP/lcGLONASS").toString() == "P3&L3") {
      opt->_lcGLONASS.push_back(t_lc::cIF);
      opt->_lcGLONASS.push_back(t_lc::lIF);
    }

    if      (settings.value("PPP/lcGalileo").toString() == "P3") {
      opt->_lcGalileo.push_back(t_lc::cIF);
    }
    else if (settings.value("PPP/lcGalileo").toString() == "L3") {
      opt->_lcGalileo.push_back(t_lc::lIF);
    }
    else if (settings.value("PPP/lcGalileo").toString() == "P3&L3") {
      opt->_lcGalileo.push_back(t_lc::cIF);
      opt->_lcGalileo.push_back(t_lc::lIF);
    }

    _options << opt;
  }
}

