// Part of BNC, a utility for retrieving decoding and
// converting GNSS data streams from NTRIP broadcasters.
//
// Copyright (C) 2007
// German Federal Agency for Cartography and Geodesy (BKG)
// http://www.bkg.bund.de
// Czech Technical University Prague, Department of Geodesy
// http://www.fsv.cvut.cz
//
// Email: euref-ip@bkg.bund.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation, version 2.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifndef GPSDECODER_H
#define GPSDECODER_H

#include <iostream>
#include <iomanip>
#include <vector>
#include <string>

#include <QtCore>

#include "bncconst.h"
#include "bnctime.h"
#include "satObs.h"
#include "crs.h"

class bncRinex;
using namespace std;

class GPSDecoder {
 public:
  GPSDecoder();
  virtual ~GPSDecoder();

  virtual t_irc Decode(char* buffer, int bufLen,
                       std::vector<std::string>& errmsg) = 0;


  virtual int corrGPSEpochTime() const {return -1;}

  void initRinex(const QByteArray& staID, const QUrl& mountPoint,
                 const QByteArray& latitude, const QByteArray& longitude,
                 const QByteArray& nmea, const QByteArray& ntripVersion);

  void dumpRinexEpoch(const t_satObs& obs, const QByteArray& format);

  void setRinexReconnectFlag(bool flag);

  struct t_typeInfo {
    t_typeInfo() {
      type = 0;
      size = 0;
    };
    int    type;
    size_t size;
  };


  struct t_antRefPoint {
    enum t_type { ARP, APC };

    t_antRefPoint() {
      xx = yy = zz = height = 0.0;
      type = ARP;
      height_f = false;
      message  = 0;
    };

    double xx;
    double yy;
    double zz;
    t_type type;
    double height;
    bool   height_f;
    int    message;
  };

  struct t_antInfo {
    t_antInfo() {
    };
    char descriptor[256];
    char serialnumber[256];
  };

  struct t_recInfo {
    t_recInfo() {
    };
    char descriptor[256];
    char serialnumber[256];
    char firmware[256];
  };

  class t_GloBiasInfo {
  public:
    t_GloBiasInfo() {
      clear();
    };
    bool operator==(const t_GloBiasInfo& biasInfo2) const {
      if (staID     == biasInfo2.staID &&
          indicator == biasInfo2.indicator &&
          L1C_value == biasInfo2.L1C_value &&
          L1P_value == biasInfo2.L1P_value &&
          L2C_value == biasInfo2.L2C_value &&
          L2P_value == biasInfo2.L2P_value  ) {
        return true;
      }
      else {
        return false;
      }
    }

    void set(const t_GloBiasInfo& biasInfo) {
      staID     = biasInfo.staID;
      indicator = biasInfo.indicator;
      L1C_value = biasInfo.L1C_value;
      L1P_value = biasInfo.L1P_value;
      L2C_value = biasInfo.L2C_value;
      L2P_value = biasInfo.L2P_value;
      changed = true;
    }

    void clear() {cout << "init" <<  endl;
      staID = 0;
      indicator = 0;
      L1C_value = 0.0;
      L1P_value = 0.0;
      L2C_value = 0.0;
      L2P_value = 0.0;
      changed = false;
    }
    QString toString() {
      QString biasIndicator =  (indicator == 1) ? QString("aligned") : QString("unaligned");
      QString biasesStr =
          QString(": GLONASS L1/L2 Code-Phase Biases: staID=%1 indicator=%2 L1C=%3 L1P=%4 L2C=%5 L2P=%6")
          .arg(staID).arg(biasIndicator)
          .arg(L1C_value, 0, 'f', 2)
          .arg(L1P_value, 0, 'f', 2)
          .arg(L2C_value, 0, 'f', 2)
          .arg(L2P_value, 0, 'f', 2);

      return biasesStr;
    }
    int      staID;
    int      indicator;
    double   L1C_value;
    double   L1P_value;
    double   L2C_value;
    double   L2P_value;
    bool     changed;


  };

  /** List of observations */
  QList<t_satObs>         _obsList;
  QList<t_typeInfo>       _typeList;           // RTCM message type as message size
  QList<t_antInfo>        _antType;            // RTCM antenna descriptor
  QList<t_recInfo>        _recType;            // RTCM receiver descriptor
  QList<t_antRefPoint>    _antList;            // RTCM antenna XYZ
  QList<t_helmertPar>     _helmertPar;         // List of Helmert parameter sets
  QList<t_serviceCrs>     _serviceCrs;         // Service CRS
  QList<t_rtcmCrs>        _rtcmCrs;            // RTCM CRS
  QString                 _gloFrq;             // GLONASS slot
  t_GloBiasInfo           _gloBiases;          // RTCM GLO bias information message
  bncRinex*               _rnx;                // RINEX writer
};

#endif
