
/* -------------------------------------------------------------------------
 * BKG NTRIP Server
 * -------------------------------------------------------------------------
 *
 * Class:      bncSinexTro
 *
 * Purpose:    writes SINEX TRO files
 *
 * Author:     A. Stürze
 *
 * Created:    19-Feb-2015
 *
 * Changes:
 *
 * -----------------------------------------------------------------------*/

#include <math.h>
#include <iomanip>

#include "bncsinextro.h"

using namespace BNC_PPP;
using namespace std;

// Constructor
////////////////////////////////////////////////////////////////////////////
bncSinexTro::bncSinexTro(const t_pppOptions* opt,
                         const QString& sklFileName, const QString& intr,
                         int sampl)
  : bncoutf(sklFileName, intr, sampl) {

  _opt   = opt;
  _sampl =  sampl;

  _antex = 0;
}

// Destructor
////////////////////////////////////////////////////////////////////////////
bncSinexTro::~bncSinexTro() {
  closeFile();
  if (_antex)
    delete _antex;
}

// Write Header
////////////////////////////////////////////////////////////////////////////
void bncSinexTro::writeHeader(const QDateTime& datTim) {
  int    GPSWeek;
  double GPSWeeks;
  bncSettings settings;
  GPSweekFromDateAndTime(datTim, GPSWeek, GPSWeeks);
  int daysec    = int(fmod(GPSWeeks, 86400.0));
  int dayOfYear = datTim.date().dayOfYear();
  QString yy    = datTim.toString("yy");
  QString creationTime = QString("%1:%2:%3").arg(yy)
                                            .arg(dayOfYear, 3, 10, QLatin1Char('0'))
                                            .arg(daysec   , 5, 10, QLatin1Char('0'));
  QString startTime = creationTime;
  QString intStr = settings.value("PPP/snxtroIntr").toString();
  int intr, indHlp = 0;
  if      ((indHlp = intStr.indexOf("min")) != -1) {
    intr = intStr.left(indHlp-1).toInt();
    intr *= 60;
  }
  else if ((indHlp = intStr.indexOf("hour")) != -1) {
    intr = intStr.left(indHlp-1).toInt();
    intr *= 3600;
  }
  else if ((indHlp = intStr.indexOf("day")) != -1) {
    intr = intStr.left(indHlp-1).toInt();
    intr *= 86400;
  }
  int nominalStartSec = daysec - (int(fmod(double(daysec), double(intr))));
  int nominalEndSec = nominalStartSec + intr - _sampl;
  QString endTime = QString("%1:%2:%3").arg(yy)
                                       .arg(dayOfYear     , 3, 10, QLatin1Char('0'))
                                       .arg(nominalEndSec , 5, 10, QLatin1Char('0'));
  int numEpochs = ((nominalEndSec - daysec) / _sampl) +1;
  QString epo    = QString("%1").arg(numEpochs, 5, 10, QLatin1Char('0'));
  QString ac     = QString("%1").arg(settings.value("PPP/snxtroAc").toString(),3,QLatin1Char(' '));
  QString solId  = settings.value("PPP/snxtroSolId").toString();
  QString corr = "";
  if      (settings.value("PPP/dataSource").toString() == "Real-Time Streams") {
    corr = settings.value("PPP/corrMount").toString();
  }
  else if (settings.value("PPP/dataSource").toString() == "RINEX Files") {
    corr = settings.value("PPP/corrFile").toString();
  }

  QString signalPriorities = QString::fromStdString(_opt->_signalPriorities);
  if (!signalPriorities.size()) {
    signalPriorities = "G:12&CWPSLX R:12&CP E:1&CBX E:5&QIX C:26&IQX";
  }
  QStringList priorList = signalPriorities.split(" ", QString::SkipEmptyParts);
  QStringList frqStrList;
  for (unsigned iFreq = 1; iFreq < t_frequency::max; iFreq++) {
    t_frequency::type frqType = static_cast<t_frequency::type>(iFreq);
    char frqSys = t_frequency::toString(frqType)[0];
    char frqNum = t_frequency::toString(frqType)[1];
    QStringList hlp;
    for (int ii = 0; ii < priorList.size(); ii++) {
      if (priorList[ii].indexOf(":") != -1) {
        hlp = priorList[ii].split(":", QString::SkipEmptyParts);
        if (hlp.size() == 2 && hlp[0].length() == 1 && hlp[0][0] == frqSys) {
          hlp = hlp[1].split("&", QString::SkipEmptyParts);
        }
        if (hlp.size() == 2 && hlp[0].indexOf(frqNum) != -1) {
          frqStrList.append(QString("%1%2").arg(frqSys).arg(frqNum));
        }
      }
    }
  }

  _out << "%=TRO 2.00 " << ac.toStdString() << " "
       << creationTime.toStdString() << " " << ac.toStdString() << " "
       << startTime.toStdString()    << " " << endTime.toStdString() << " P "
       << epo.toStdString() << " 0 " << " T " << endl;
  _out << "*-------------------------------------------------------------------------------" << endl;
  _out << "+FILE/REFERENCE" << endl;
  _out << "*INFO_TYPE_________ INFO________________________________________________________" << endl;
  _out << " DESCRIPTION        " << "BNC generated SINEX TRO file" << endl;
  _out << " OUTPUT             " << "Total Troposphere Zenith Path Delay Product" << endl;
  _out << " SOFTWARE           " <<  BNCPGMNAME <<  endl;
  _out << " INPUT              " << "OBS: "    << _opt->_roverName.substr(0,9) << ", SSR: "  << corr.toStdString() << endl;
  _out << " VERSON NUMBER      " <<  QString("%1").arg(solId, 10, QLatin1Char(' ')).toStdString() << endl;
  _out << "-FILE/REFERENCE" << endl;
  _out << "*-------------------------------------------------------------------------------" << endl;


  QString blqFileName = QString::fromStdString(_opt->_blqFileName);
  QString blqFileBaseName;
  QString blqFileExtension;
  if (! blqFileName.isEmpty()) {
    QFileInfo fileInfo(blqFileName);
    blqFileBaseName  = fileInfo.baseName();
    blqFileExtension = fileInfo.completeSuffix();
    if (!blqFileExtension.isEmpty()) {
      blqFileExtension = "." + blqFileExtension;
    }
  }
  _out << "+TROP/DESCRIPTION" << endl;
  _out << "*KEYWORD______________________ VALUE(S)______________" << endl;
  _out << " SAMPLING INTERVAL                               "
       << setw(4) << _sampl << endl;
  _out << " SAMPLING TROP                                   "
       << setw(4) << _sampl << endl;
  _out << " ELEVATION CUTOFF ANGLE                          "
       << setw(4) <<  int(_opt->_minEle * 180.0/M_PI) << endl;
  if (! blqFileName.isEmpty()) {
    _out << " OCEAN TIDE LOADING MODEL      " << blqFileBaseName.toStdString() + blqFileExtension.toStdString() << endl;
  }
  _out << " TROP MAPPING FUNCTION         " << "Saastamoinen" << endl;
  _out << " SOLUTION_FIELDS_1             " << "TROTOT STDEV" << endl;
  _out << "-TROP/DESCRIPTION"<< endl;
  _out << "*-------------------------------------------------------------------------------" << endl;

  double recEll[3];
  int lonD, lonM,  latD, latM;
  double lonS, latS;
  xyz2ell(_opt->_xyzAprRover.data(), recEll);
  deg2DMS(recEll[0] * 180.0 / M_PI, latD, latM, latS);
  deg2DMS(recEll[1] * 180.0 / M_PI, lonD, lonM, lonS);
  QString country;
  QListIterator<QString> it(settings.value("mountPoints").toStringList());
  while (it.hasNext()) {
    QStringList hlp = it.next().split(" ");
    if (hlp.size() < 7)
      continue;
    if (hlp.join(" ").indexOf(QString::fromStdString(_opt->_roverName), 0) != -1) {
      country = hlp[2];
    }
  }
  _out << "+SITE/ID" << endl;
  _out << "*STATION__ PT __DOMES__ T _STATION_DESCRIPTION__ _LONGITUDE _LATITUDE_ _HGT_ELI_" << endl;
  _out << " " << _opt->_roverName.substr(0,9) << "  A           P "
       << country.toStdString() << "                   "
       << QString(" %1").arg(recEll[0]* 180.0 / M_PI,10, 'f', 6, QLatin1Char(' ')).toStdString()
       << QString(" %1").arg(recEll[1]* 180.0 / M_PI,10, 'f', 6, QLatin1Char(' ')).toStdString()
       << QString(" %1").arg(recEll[2], 9, 'f', 3, QLatin1Char(' ')).toStdString()
       << endl;
  _out << "-SITE/ID" << endl;
  _out << "*-------------------------------------------------------------------------------" << endl;

  _out << "+TROP/COORDINATES" << endl;
  _out << "*SITE      PT SOLN T STA_X_______ STA_Y_______ STA_Z_______ SYSTEM REMARK" << endl;
  _out << " " << _opt->_roverName.substr(0,9) << "  A "
       <<  solId.toStdString() << " P"
       << QString(" %1").arg(_opt->_xyzAprRover(1), 12, 'f', 3, QLatin1Char(' ')).toStdString()
       << QString(" %1").arg(_opt->_xyzAprRover(2), 12, 'f', 3, QLatin1Char(' ')).toStdString()
       << QString(" %1").arg(_opt->_xyzAprRover(3), 12, 'f', 3, QLatin1Char(' ')).toStdString()
       << " IGS20 " << ac.toStdString() << endl;
  _out << "-TROP/COORDINATES"<< endl;
  _out << "*-------------------------------------------------------------------------------" << endl;


  _out << "+SITE/ECCENTRICITY" << endl;
  _out << "*                                             UP______ NORTH___ EAST____" << endl;
  _out << "*SITE      PT SOLN T DATA_START__ DATA_END____ AXE ARP->BENCHMARK(M)_________" << endl;
  _out << " " << _opt->_roverName.substr(0,9) << "  A "
       <<  solId.toStdString() << " P "
       << startTime.toStdString() << " " << endTime.toStdString() << " UNE"
       << QString("%1").arg(_opt->_neuEccRover(3), 9, 'f', 4, QLatin1Char(' ')).toStdString()
       << QString("%1").arg(_opt->_neuEccRover(1), 9, 'f', 4, QLatin1Char(' ')).toStdString()
       << QString("%1").arg(_opt->_neuEccRover(2), 9, 'f', 4, QLatin1Char(' ')).toStdString() << endl;
  _out << "-SITE/ECCENTRICITY" << endl;
  _out << "*-------------------------------------------------------------------------------" << endl;

  if (!_opt->_recNameRover.empty()) {
    _out << "+SITE/RECEIVER" << endl;
    _out << "*STATION__ PT SOLN T __DATA_START__ __DATA_END____ DESCRIPTION_________ S/N_________________ FIRMW______" << endl;
    _out << " " << _opt->_roverName.substr(0,9) << "  A "  <<  solId.toStdString() << " P "
         << startTime.toStdString() << " " << endTime.toStdString()
         << QString(" %1").arg(_opt->_recNameRover.c_str(), 20,QLatin1Char(' ')).toStdString()
         << " -----" << " -----------" << endl;
    _out << "-SITE/RECEIVER" << endl;
    _out << "*-------------------------------------------------------------------------------" << endl;
  }

  _out << "+SITE/ANTENNA" << endl;
  _out << "*STATION__ PT SOLN T __DATA_START__ __DATA_END____ DESCRIPTION_________ S/N_________________ PCV_MODEL_" << endl;
  _out << " " << _opt->_roverName.substr(0,9) << "  A "  <<  solId.toStdString() << " P "
       << startTime.toStdString() << " " << endTime.toStdString() << " "
       << _opt->_antNameRover << " -----" << endl;
  _out << "-SITE/ANTENNA" << endl;
  _out << "*-------------------------------------------------------------------------------" << endl;

  if (!_opt->_antexFileName.empty()) {
    _antex = new bncAntex(_opt->_antexFileName.c_str());
    if (_opt->_LCsGPS.size()) {
      _out << "+SITE/GPS_PHASE_CENTER" << endl;
      _out << "*                           UP____ NORTH_ EAST__ UP____ NORTH_ EAST__" << endl;
      _out << "*DESCRIPTION_________ S/N__ L1->ARP(m)__________ L2->ARP(m)__________ AZ_EL____" << endl;
      _out << QString(" %1").arg(_opt->_antNameRover.c_str(), 20,QLatin1Char(' ')).toStdString()
           <<  " -----";
           for (int i = 0; i < frqStrList.size(); ++i) {
             if (frqStrList.at(i).contains('G')) {//cout << frqStrList.at(i).toStdString() << endl;
               _out << _antex->pcoSinexString(_opt->_antNameRover, t_frequency::toInt(frqStrList.at(i).toStdString())).toStdString();
             }
           }
      _out << " ---------" << endl;
      _out << "-SITE/GPS_PHASE_CENTER" << endl;
      _out << "*-------------------------------------------------------------------------------" << endl;
    }
    if (_opt->_LCsGLONASS.size()) {
      _out << "+SITE/GLONASS_PHASE_CENTER" << endl;
      _out << "*                           UP____ NORTH_ EAST__ UP____ NORTH_ EAST__" << endl;
      _out << "*DESCRIPTION_________ S/N__ L1->ARP(m)__________ L2->ARP(m)__________ AZ_EL____" << endl;
      _out << QString(" %1").arg(_opt->_antNameRover.c_str(), 20,QLatin1Char(' ')).toStdString()
           <<  " -----";
           for (int i = 0; i < frqStrList.size(); ++i) {
             if (frqStrList.at(i).contains('R')) {//cout << frqStrList.at(i).toStdString() << endl;
               _out << _antex->pcoSinexString(_opt->_antNameRover, t_frequency::toInt(frqStrList.at(i).toStdString())).toStdString();
             }
           }
      _out << " ---------" << endl;
      _out << "-SITE/GLONASS_PHASE_CENTER" << endl;
      _out << "*-------------------------------------------------------------------------------" << endl;
    }
    if (_opt->_LCsGalileo.size()) {
      _out << "+SITE/GALILEO_PHASE_CENTER" << endl;
      _out << "*                           UP____ NORTH_ EAST__ UP____ NORTH_ EAST__" << endl;
      _out << "*DESCRIPTION_________ S/N__ L1->ARP(m)__________ L2->ARP(m)__________ AZ_EL____" << endl;
      _out << QString(" %1").arg(_opt->_antNameRover.c_str(), 20,QLatin1Char(' ')).toStdString()
           <<  " -----";
           for (int i = 0; i < frqStrList.size(); ++i) {
             if (frqStrList.at(i).contains('E')) {//cout << frqStrList.at(i).toStdString() << endl;
               _out << _antex->pcoSinexString(_opt->_antNameRover, t_frequency::toInt(frqStrList.at(i).toStdString())).toStdString();
             }
           }
      _out << " ---------" << endl;
      _out << "-SITE/GALILEO_PHASE_CENTER" << endl;
      _out << "*-------------------------------------------------------------------------------" << endl;
    }
    if (_opt->_LCsBDS.size()) {
      _out << "+SITE/BEIDOU_PHASE_CENTER" << endl;
      _out << "*                           UP____ NORTH_ EAST__ UP____ NORTH_ EAST__" << endl;
      _out << "*DESCRIPTION_________ S/N__ L1->ARP(m)__________ L2->ARP(m)__________ AZ_EL____" << endl;
      _out << QString(" %1").arg(_opt->_antNameRover.c_str(), 20,QLatin1Char(' ')).toStdString()
               <<  " -----";
               for (int i = 0; i < frqStrList.size(); ++i) {
                 if (frqStrList.at(i).contains('C')) {//cout << frqStrList.at(i).toStdString() << endl;
                   _out << _antex->pcoSinexString(_opt->_antNameRover, t_frequency::toInt(frqStrList.at(i).toStdString())).toStdString();
                 }
               }
      _out << " ---------" << endl;
      _out << "-SITE/BEIDOU_PHASE_CENTER" << endl;
      _out << "*-------------------------------------------------------------------------------" << endl;
    }
    delete _antex;
    _antex = 0;
  }

  _out << "+TROP/SOLUTION" << endl;
  _out << "*STATION__ ____EPOCH_____ TROTOT STDDEV " << endl;
}

// Write One Epoch
////////////////////////////////////////////////////////////////////////////
t_irc bncSinexTro::write(QByteArray staID, int GPSWeek, double GPSWeeks,
    double trotot, double stdev) {

  QDateTime datTim = dateAndTimeFromGPSweek(GPSWeek, GPSWeeks);
  int daysec    = int(fmod(GPSWeeks, 86400.0));
  int dayOfYear = datTim.date().dayOfYear();
  QString yy    = datTim.toString("yy");
  QString time  = QString("%1:%2:%3").arg(yy)
                                     .arg(dayOfYear, 3, 10, QLatin1Char('0'))
                                     .arg(daysec   , 5, 10, QLatin1Char('0'));

  if ((reopen(GPSWeek, GPSWeeks) == success) &&
      (fmod(daysec, double(_sampl)) == 0.0)) {
    _out << ' '  << staID.left(9).data() << ' ' << time.toStdString() << ' '
         << noshowpos << setw(6) << setprecision(1) << trotot * 1000.0
         << noshowpos << setw(6) << setprecision(1) << stdev  * 1000.0 << endl;
    _out.flush();
    return success;
  }  else {
    return failure;
  }
}

// Close File (write last lines)
////////////////////////////////////////////////////////////////////////////
void bncSinexTro::closeFile() {
  _out << "-TROP/SOLUTION" << endl;
  _out << "%=ENDTROP" << endl;
  bncoutf::closeFile();
}




