
// Part of BNC, a utility for retrieving decoding and
// converting GNSS data streams from NTRIP broadcasters.
//
// Copyright (C) 2007
// German Federal Agency for Cartography and Geodesy (BKG)
// http://www.bkg.bund.de
// Czech Technical University Prague, Department of Geodesy
// http://www.fsv.cvut.cz
//
// Email: euref-ip@bkg.bund.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation, version 2.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

/* -------------------------------------------------------------------------
 * BKG NTRIP Client
 * -------------------------------------------------------------------------
 *
 * Class:      t_pppThread, t_pppRun
 *
 * Purpose:    Single PPP Client (running in its own thread)
 *
 * Author:     L. Mervart
 *
 * Created:    29-Jul-2014
 *
 * Changes:    
 *
 * -----------------------------------------------------------------------*/


#include <iostream>
#include <iomanip>
#include <string.h>

#include "pppThread.h"
#include "bnccore.h"

using namespace BNC;
using namespace std;

// Constructor
////////////////////////////////////////////////////////////////////////////
t_pppThread::t_pppThread(const t_options* opt) : QThread(0) {
  _opt   = opt;
  _pppRun = 0;
  connect(this, SIGNAL(finished()), this, SLOT(deleteLater()));

  connect(this, SIGNAL(newMessage(QByteArray,bool)), 
          BNC_CORE, SLOT(slotMessage(const QByteArray,bool)));
}

// Destructor
////////////////////////////////////////////////////////////////////////////
t_pppThread::~t_pppThread() {
  cout << "~t_pppThread" << endl;
  delete _pppRun;
}

// Run (virtual)
////////////////////////////////////////////////////////////////////////////
void t_pppThread::run() {
  try {
    _pppRun = new t_pppRun(_opt);
    QThread::exec();
  }
  catch (pppExcept exc) {
    _pppRun = 0;
    emit newMessage(QByteArray(exc.what().c_str()), true);
  }
}

// Constructor
////////////////////////////////////////////////////////////////////////////
t_pppRun::t_pppRun(const t_options* opt) {
  _opt = opt;
  connect(this, SIGNAL(newMessage(QByteArray,bool)), 
          BNC_CORE, SLOT(slotMessage(const QByteArray,bool)));
  if (_opt->_realTime) {
    connect(BNC_CORE->caster(), SIGNAL(newObs(QByteArray, QList<t_obs>)),
            this, SLOT(slotNewObs(QByteArray, QList<t_obs>)));

    connect(BNC_CORE, SIGNAL(newEphGPS(gpsephemeris)),
            this, SLOT(slotNewEphGPS(gpsephemeris)));
  
    connect(BNC_CORE, SIGNAL(newEphGlonass(glonassephemeris)),
            this, SLOT(slotNewEphGlonass(glonassephemeris)));
  
    connect(BNC_CORE, SIGNAL(newEphGalileo(galileoephemeris)),
            this, SLOT(slotNewEphGalileo(galileoephemeris)));

    connect(BNC_CORE, SIGNAL(newCorrections(QStringList)),
            this, SLOT(slotNewCorrections(QStringList)));

    _pppClient = new t_pppClient(_opt);
  }
  else {
    throw pppExcept("t_pppRun: post-processing not yet implemented");
  }
}

// Destructor
////////////////////////////////////////////////////////////////////////////
t_pppRun::~t_pppRun() {
  cout << "~t_pppRun" << endl;
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewEphGPS(gpsephemeris gpseph) {
  QMutexLocker locker(&_mutex);
  t_ephGPS eph;
  eph.set(&gpseph);
  _pppClient->putEphemeris(&eph);
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewEphGlonass(glonassephemeris gloeph) {
  QMutexLocker locker(&_mutex);
  t_ephGlo eph;
  eph.set(&gloeph);
  _pppClient->putEphemeris(&eph);
}
  
// 
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewEphGalileo(galileoephemeris galeph) {
  QMutexLocker locker(&_mutex);
  t_ephGal eph;
  eph.set(&galeph);
  _pppClient->putEphemeris(&eph);
}

// 
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewCorrections(QStringList corrList) {
  QMutexLocker locker(&_mutex);

  QStringListIterator it(corrList);
  while (it.hasNext()) {
    it.next();
  }

  // _pppClient->putOrbCorrections(const std::vector<t_orbCorr>& corr); 
  // _pppClient->putClkCorrections(const std::vector<t_clkCorr>& corr); 
  // _pppClient->putBiases(const std::vector<t_satBiases>& biases);   
}
//
////////////////////////////////////////////////////////////////////////////
void t_pppRun::slotNewObs(QByteArray staID, QList<t_obs> obsList) {
  QMutexLocker locker(&_mutex);

  if (string(staID.data()) != _opt->_roverName) {
    return;
  }

  vector<t_pppSatObs> satObs;
  t_output            output;

  _pppClient->processEpoch(satObs, &output);

  emit newMessage(QByteArray(output._log.c_str()), true);
}
    
