// Part of BNC, a utility for retrieving decoding and
// converting GNSS data streams from NTRIP broadcasters.
//
// Copyright (C) 2007
// German Federal Agency for Cartography and Geodesy (BKG)
// http://www.bkg.bund.de
// Czech Technical University Prague, Department of Geodesy
// http://www.fsv.cvut.cz
//
// Email: euref-ip@bkg.bund.de
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation, version 2.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

/* -------------------------------------------------------------------------
 * BKG NTRIP Client
 * -------------------------------------------------------------------------
 *
 * Class:      t_rnxNavFile
 *
 * Purpose:    Reads RINEX Navigation File
 *
 * Author:     L. Mervart
 *
 * Created:    24-Jan-2012
 *
 * Changes:    
 *
 * -----------------------------------------------------------------------*/

#include <iostream>
#include <newmatio.h>
#include "rnxnavfile.h"
#include "bncapp.h"
#include "bncutils.h"
#include "RTCM3/ephemeris.h"

using namespace std;

// Constructor
////////////////////////////////////////////////////////////////////////////
t_rnxNavFile::t_rnxNavHeader::t_rnxNavHeader() {
  _version = 0.0;
  _glonass = false;
}

// Destructor
////////////////////////////////////////////////////////////////////////////
t_rnxNavFile::t_rnxNavHeader::~t_rnxNavHeader() {
}

// Read Header
////////////////////////////////////////////////////////////////////////////
t_irc t_rnxNavFile::t_rnxNavHeader::read(QTextStream* stream) {
  while (stream->status() == QTextStream::Ok && !stream->atEnd()) {
    QString line = stream->readLine();
    if (line.isEmpty()) {
      continue;
    }
    QString value = line.left(60).trimmed();
    QString key   = line.mid(60).trimmed();
    if      (key == "END OF HEADER") {
      break;
    }
    else if (key == "RINEX VERSION / TYPE") {
      QTextStream in(value.toAscii(), QIODevice::ReadOnly);
      in >> _version;
      if (value.indexOf("GLONASS") != -1) {
        _glonass = true;
      }
    }
  }

  return success;
}

// Constructor
////////////////////////////////////////////////////////////////////////////
t_rnxNavFile::t_rnxNavFile(const QString& fileName, e_inpOut inpOut) {
  _inpOut = inpOut;
  _stream = 0;
  _file   = 0;
  if (_inpOut == input) {
    openRead(fileName);
  }
  else {
    openWrite(fileName);
  }
}

// Open for input
////////////////////////////////////////////////////////////////////////////
void t_rnxNavFile::openRead(const QString& fileName) {

  _fileName = fileName; expandEnvVar(_fileName);
  _file     = new QFile(_fileName);
  _file->open(QIODevice::ReadOnly | QIODevice::Text);
  _stream = new QTextStream();
  _stream->setDevice(_file);

  _header.read(_stream);
  this->read(_stream);
}

// Open for output
////////////////////////////////////////////////////////////////////////////
void t_rnxNavFile::openWrite(const QString& fileName) {

  _fileName = fileName; expandEnvVar(_fileName);
  _file     = new QFile(_fileName);
  _file->open(QIODevice::WriteOnly | QIODevice::Text);
  _stream = new QTextStream();
  _stream->setDevice(_file);
}

// Destructor
////////////////////////////////////////////////////////////////////////////
t_rnxNavFile::~t_rnxNavFile() {
  close();
  for (unsigned ii = 0; ii < _ephs.size(); ii++) {
    delete _ephs[ii];
  }  
}

// Close
////////////////////////////////////////////////////////////////////////////
void t_rnxNavFile::close() {
  delete _stream; _stream = 0;
  delete _file;   _file = 0;
}

// Read File Content
////////////////////////////////////////////////////////////////////////////
void t_rnxNavFile::read(QTextStream* stream) {

  while (stream->status() == QTextStream::Ok && !stream->atEnd()) {
    QString line = stream->readLine();
    if (line.isEmpty()) {
      continue;
    }
    QStringList hlp = line.split(QRegExp("\\s+"), QString::SkipEmptyParts);
    QString prn;
    if (version() >= 3.0) {
      prn = hlp.at(0);
    }
    else {
      if (glonass()) {
        prn = QString("R%1").arg(hlp.at(0).toInt(), 2, 10, QChar('0'));
      }
      else {
        prn = QString("G%1").arg(hlp.at(0).toInt(), 2, 10, QChar('0'));
      }
    }
    t_eph* eph = 0;
    QStringList lines; lines << line;
    if      (prn[0] == 'G') {
      for (int ii = 1; ii < 8; ii++) {
        lines << stream->readLine();
      }
      eph = new t_ephGPS(version(), lines);
    }
    else if (prn[0] == 'R') {
      for (int ii = 1; ii < 4; ii++) {
        lines << stream->readLine();
      }
      eph = new t_ephGlo(version(), lines);
    }
    else if (prn[0] == 'E') {
      for (int ii = 1; ii < 8; ii++) {
        lines << stream->readLine();
      }
      eph = new t_ephGal(version(), lines);
    }
    if (eph && eph->ok()) {
      _ephs.push_back(eph);
    }
    else {
      delete eph;
    }
  }
}

// Read Next Ephemeris
////////////////////////////////////////////////////////////////////////////
t_eph* t_rnxNavFile::getNextEph(const bncTime& tt, 
                                const QMap<QString, int>* corrIODs) {

  // Get Ephemeris according to IOD
  // ------------------------------
  if (corrIODs) {
    QMapIterator<QString, int> itIOD(*corrIODs);
    while (itIOD.hasNext()) {
      itIOD.next();
      QString prn = itIOD.key();
      int     iod = itIOD.value();
      vector<t_eph*>::iterator it = _ephs.begin();
      while (it != _ephs.end()) {
        t_eph* eph = *it;
        double dt = eph->TOC() - tt;
        if (dt < 8*3600.0 && eph->prn() == prn && eph->IOD() == iod) {
          it = _ephs.erase(it);
          return eph;
        }
        ++it;
      }
    }
  }

  // Get Ephemeris according to time
  // -------------------------------
  else {
    vector<t_eph*>::iterator it = _ephs.begin();
    while (it != _ephs.end()) {
      t_eph* eph = *it;

      double dt = eph->TOC() - tt;

      if (dt < 2*3600.0) {
        it = _ephs.erase(it);
        return eph;
      }
      ++it;
    }
  }

  return 0;
}

// 
////////////////////////////////////////////////////////////////////////////
void t_rnxNavFile::writeHeader(const QMap<QString, QString>* txtMap) {

  bncApp* app = (bncApp*) qApp;

  QString     runBy = app->userName();
  QStringList comments;

  if (txtMap) {
    QMapIterator<QString, QString> it(*txtMap);
    while (it.hasNext()) {
      it.next();
      if      (it.key() == "RUN BY") {
        runBy = it.value();
      }
      else if (it.key() == "COMMENT") {
        comments = it.value().split("\\n", QString::SkipEmptyParts);
      }
    }
  }

  if (version() < 3.0) {
    *_stream << QString("%1           Navigation data")
      .arg(_header._version, 9, 'f', 2)
      .leftJustified(60)
             << "RINEX VERSION / TYPE\n";
  }
  else {
    *_stream << QString("%1           Navigation data     Mixed")
      .arg(_header._version, 9, 'f', 2)
      .leftJustified(60)
             << "RINEX VERSION / TYPE\n";
  }

  *_stream << QString("%1%2%3")
    .arg(app->pgmName(), -20)
    .arg(runBy.trimmed().left(20), -20)
    .arg(QDateTime::currentDateTime().toUTC().toString("yyyyMMdd hhmmss UTC"), -20)
    .leftJustified(60)
           << "PGM / RUN BY / DATE\n";

  *_stream << QString()
    .leftJustified(60)
           << "END OF HEADER\n";
}

// 
////////////////////////////////////////////////////////////////////////////
void t_rnxNavFile::writeEph(const t_eph* eph) {
  *_stream << eph->toString(version());
}
